
library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;

entity fifo_dram_sync is
   generic (
      WIDTH : INTEGER := 16;
      DEPTH_BITS : INTEGER := 5;
      RAM_STYLE_HINT : STRING := "auto"
  );
   port (
      clk      : IN std_logic;
      i_din    : IN std_logic_VECTOR(7 downto 0);
      i_rd_en  : IN std_logic;
      i_wr_en  : IN std_logic;
      o_dout   : OUT std_logic_VECTOR(7 downto 0);
      o_empty  : OUT std_logic; -- one clock delay from read strobe
      o_full   : OUT std_logic; -- one clock delay from write strobe
      o_almost_full : OUT std_logic; -- one clock delay from write stobe, 1 from full
      o_almost_empty : OUT std_logic; -- one clock delay from read strobe, 1 from empty
      i_reset  : IN std_logic := '0';
      o_count  : OUT std_logic_vector(DEPTH_BITS downto 0)
   );
end fifo_dram_sync;

architecture rtl of fifo_dram_sync is

attribute RAM_STYLE : string;
type ram_type is array((2**DEPTH_BITS)-1 downto 0) of std_logic_vector(WIDTH-1 downto 0);
signal RAM : ram_type := (others=>(others=>'0')); 
attribute RAM_STYLE of RAM: signal is RAM_STYLE_HINT;

signal write_addr : std_logic_vector(DEPTH_BITS-1 downto 0) := (others=>'0');
signal read_addr  : std_logic_vector(DEPTH_BITS-1 downto 0) := (others=>'0');
signal count      : std_logic_vector(DEPTH_BITS   downto 0) := (others=>'0');
signal full       : std_logic;
signal empty      : std_logic;

begin

full <= '1' when count = CONV_STD_LOGIC_VECTOR(2**DEPTH_BITS, DEPTH_BITS+1) else '0';
empty <= '1' when count = CONV_STD_LOGIC_VECTOR(0, DEPTH_BITS+1) else '0';

ramblock : process (clk)
begin
   if rising_edge(clk) then
      if i_wr_en='1' and full='0' then
         RAM(conv_integer(write_addr)) <= i_din;
      end if;
   end if;
end process ramblock;

o_dout  <= RAM(conv_integer(read_addr));
o_empty <= empty;
o_full  <= full;
o_almost_full  <= '1' when full='1' or count(DEPTH_BITS-1 downto 0) = CONV_STD_LOGIC_VECTOR(2**DEPTH_BITS-1, DEPTH_BITS) else '0';
o_almost_empty <= '1' when empty='1' or ((count<=CONV_STD_LOGIC_VECTOR(1, DEPTH_BITS+1)) and i_rd_en='1') else '0';
o_count <= count;

address : process (clk, i_reset)
begin
   if i_reset='1' then
      write_addr <= (others=>'0');
      read_addr <= (others=>'0');
   elsif rising_edge(clk) then
      if i_wr_en='1' and full='0' then
         write_addr <= write_addr+'1';
      end if;
      if i_rd_en='1' and empty='0' then
         read_addr <= read_addr+'1';
      end if;
   end if;
end process address;

level : process(clk, i_reset)
begin
   if i_reset='1' then
      count <= (others=>'0');
   elsif rising_edge(clk) then
      if i_wr_en='1' and i_rd_en='0' and full='0' then
         count <= count+'1';
      elsif i_wr_en='0' and i_rd_en='1' and empty='0' then
         count <= count-'1';
      end if;
   end if;
end process level;

end rtl;
