--- Title: MityDSP_L138_top.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 10/01/2010
--- Version: 1.02
---
--- Revision History
---   1.00 - Initial revision
---   1.01 - Updated for changes to EMIFA_iface and base_module.
---   1.02 - Updated for byte-access fix in EMIFA_iface.
---   1.03 - Updated to support Arria-7 replacement

library WORK;
library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;
use WORK.MityDSP_L138_pkg.ALL;

entity MityDSP_L138_top is
   generic (
      FPGA_FAMILY : string := "Spartan6";  -- Artix7 or Spartan6
      DECODE_BITS : integer range 1 to 9 := 4
   );
   port (
      i_ema_clk     : in std_logic; -- 100 MHz EMIF clock

      -- DSP EMIFA BUS Interface
      i_ema_cs0_n   : in std_logic; -- RESERVED for SDRAM Controller
      i_ema_cs2_n   : in std_logic; -- Reserved core space, not used
      i_ema_cs3_n   : in std_logic; -- NAND FLASH space, not used
      i_ema_cs4_n   : in std_logic; -- DSP core CE space
      i_ema_cs5_n   : in std_logic; -- ARM core CE space
      i_ema_oe_n    : in std_logic;
      i_ema_we_n    : in std_logic;
      io_ema_wait_n : inout std_logic_vector(1 downto 0);
      io_ema_d      : inout std_logic_vector(15 downto 0);
      i_ema_a       : in std_logic_vector(13 downto 0);
      i_ema_ba      : in std_logic_vector(1 downto 0);
      i_ema_wen_dqm : in std_logic_vector(1 downto 0); 
      i_ema_cas     : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_ras     : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_sdcke   : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_rnw     : in std_logic; -- reserved for SDRAM controller, not used

      -- DSP IRQ lines
      o_int         : out std_logic_vector(1 downto 0);
      o_nmi_n       : out std_logic
  );
end MityDSP_L138_top;

--------------------------------------------------------------------------
-- ARCHITECTURE
--------------------------------------------------------------------------
architecture rtl of MityDSP_L138_top is

constant FPGA_APPLICATION_ID: std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 00, 8);
constant FPGA_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 4);
constant FPGA_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 03, 4);
constant FPGA_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 21, 5);
constant FPGA_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 11, 4);
constant FPGA_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 16, 5);

constant CPU_ARM: integer := 0; -- ARM CPU enumeration
constant CPU_DSP: integer := 1; -- DSP CPU enumeration

constant INT0_CPU: integer := CPU_ARM; -- INT0 to CPU mapping
constant INT1_CPU: integer := CPU_DSP; -- INT1 to CPU mapping

constant CORE_BASE_MODULE:     integer := 0;
constant CORE_BASE_MODULE_LVL: integer := 0;
constant CORE_BASE_MODULE_VEC: integer := 0;

--------------------------------------------------------------------------
-- Component Declarations
--------------------------------------------------------------------------
component EMIFA_iface
   generic ( 
      DECODE_BITS   : integer range 1 to 9 := 5;
      CONFIG        : string := "UNKNOWN"        -- "MityDSP_L138" 
   );
   port (
      ema_clk       : in  std_logic; -- conditioned by a DCM
      
      -- EMIFA direct-connect signals
      i_ema_cs0_n   : in  std_logic; -- Reserved for SDRAM controller, not used
      i_ema_cs2_n   : in  std_logic; -- Reserved core space, not used
      i_ema_cs3_n   : in  std_logic; -- NAND FLASH space, not used
      i_ema_cs4_n   : in  std_logic; -- CPU 1 core space 
      i_ema_cs5_n   : in  std_logic; -- CPU 0 core space
      i_ema_oe_n    : in  std_logic;
      i_ema_we_n    : in  std_logic;
      o_ema_wait_n  : out std_logic_vector(1 downto 0);
      t_ema_wait    : out std_logic;
      i_ema_d       : in  std_logic_vector(15 downto 0);
      o_ema_d       : out std_logic_vector(15 downto 0);
      t_ema_d       : out std_logic;
      i_ema_a       : in  std_logic_vector(13 downto 0);
      i_ema_ba      : in  std_logic_vector(1 downto 0);
      i_ema_wen_dqm : in  std_logic_vector(1 downto 0); 
      i_ema_cas     : in  std_logic; -- reserved for SDRAM controller, not used
      i_ema_ras     : in  std_logic; -- reserved for SDRAM controller, not used
      i_ema_sdcke   : in  std_logic; -- reserved for SDRAM controller, not used
      i_ema_rnw     : in  std_logic; -- reserved for SDRAM controller, not used
      
      -- FPGA core interface signals
      o_core_be     : out std_logic_vector(1 downto 0);
      o_core_addr   : out std_logic_vector(5 downto 0);
      o_core_cs5    : out std_logic_vector((2**DECODE_BITS)-1 downto 0);
      o_core_cs4    : out std_logic_vector((2**DECODE_BITS)-1 downto 0);
      o_core_edi    : out std_logic_vector(15 downto 0);
      i_core_edo5   : in  bus16_vector((2**DECODE_BITS)-1 downto 0);
      i_core_edo4   : in  bus16_vector((2**DECODE_BITS)-1 downto 0);
      o_core_rd     : out std_logic;
      o_core_wr     : out std_logic
   );
end component;

component base_module
   generic (
      CONFIG       : string := "UNKNOWN"; -- "MityDSP_L138", "MityARM_1808"
      IRQ0_CPU     : integer range 0 to 1 := 0; -- IRQ0 CPU enumeration.  On L138, 0 is ARM and 1 is DSP.
      IRQ1_CPU     : integer range 0 to 1 := 1; -- IRQ1 CPU enumeration.  On L138, 0 is ARM and 1 is DSP.
      GEN_DCM_RST  : boolean := FALSE; -- Enables generation of DCM reset logic.  Uses STARTUP_SPARTAN6 primitive.
      GEN_DNA_PORT : boolean := FALSE  -- Enables generation of Device DNA read logic.  Uses DNA_PORT primitive.
   );
   port (
      ema_clk         : in  std_logic;
      i_cs            : in  std_logic;
      i_ID            : in  std_logic_vector(7 downto 0);    -- assigned Application ID number, 0xFF if unassigned
      i_version_major : in  std_logic_vector(3 downto 0);    -- major version number 1-15
      i_version_minor : in  std_logic_vector(3 downto 0);    -- minor version number 0-15
      i_year          : in  std_logic_vector(4 downto 0);    -- year since 2000
      i_month         : in  std_logic_vector(3 downto 0);    -- month (1-12)
      i_day           : in  std_logic_vector(4 downto 0);    -- day (1-32)
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_be_r          : in  std_logic_vector(1 downto 0);
      
      i_irq_map       : in  bus16_vector(1 downto 0) := (others=>(others=>'0'));
      o_irq_output    : out std_logic_vector(1 downto 0);
      
      i_dcm_status    : in  std_logic_vector(2 downto 0);  -- hook to EMIF DCM status lines
      i_dcm_lock      : in  std_logic;                     -- hook to EMIF DCM lock line
      o_dcm_reset     : out std_logic                      -- hook to EMIF DCM reset line
   );
end component;

component EMIFA_dcm
   port (
      -- Clock in ports
      CLK_IN1           : in     std_logic;
      -- Clock out ports
      CLK_OUT1          : out    std_logic;
      -- Status and control signals
      RESET             : in     std_logic;
      STATUS            : out    std_logic_vector(2 downto 0);
      INPUT_CLK_STOPPED : out    std_logic;
      LOCKED            : out    std_logic;
      CLK_VALID         : out    std_logic
   );
end component;

-- Clock/DCM signals
signal ema_clk: std_logic;
signal dcm_reset: std_logic;
signal dcm_status: std_logic_vector(2 downto 0);
signal dcm_clkstopped: std_logic;
signal dcm_locked: std_logic;
signal dcm_clkvalid: std_logic;

-- EMIFA interface signals
signal o_ema_d : std_logic_vector(15 downto 0);
signal i_ema_d : std_logic_vector(15 downto 0);
attribute keep : string;
attribute keep of o_ema_d : signal is "soft";
attribute keep of i_ema_d : signal is "soft";
signal ema_wait_n : std_logic_vector(1 downto 0) := "11";
signal t_ema_wait : std_logic;
signal t_ema_d : std_logic;
signal be_r : std_logic_vector(1 downto 0);
signal addr_r : std_logic_vector(5 downto 0);
signal cs5_r : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal cs4_r : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal edi_r : std_logic_vector(15 downto 0);
signal edo5 : bus16_vector((2**DECODE_BITS)-1 downto 0) := (others=>(others=>'0'));
signal edo4 : bus16_vector((2**DECODE_BITS)-1 downto 0) := (others=>(others=>'0'));
signal rd_r : std_logic;
signal wr_r : std_logic;
signal irq_map : bus16_vector(1 downto 0) := (others=>(others=>'0'));
signal clkfb : std_logic := '0';

begin -- architecture: rtl

--------------------------------------------------------------------------
-- System DCM Instantiation
--------------------------------------------------------------------------
gen_dcm : if FPGA_FAMILY = "Spartan6" generate
begin
   emaclkdcm : EMIFA_dcm
   port map (
      CLK_IN1            => i_ema_clk,
      CLK_OUT1           => ema_clk,
      RESET              => dcm_reset,
      STATUS             => dcm_status,
      INPUT_CLK_STOPPED  => dcm_clkstopped,
      LOCKED             => dcm_locked,
      CLK_VALID          => dcm_clkvalid
   );
end generate gen_dcm;

gen_mmcm : if FPGA_FAMILY = "Artix7" generate
begin
   -- MMCME2_BASE: Base Mixed Mode Clock Manager
   --              Artix-7
   -- Xilinx HDL Language Template, version 2021.2

   MMCME2_BASE_inst : MMCME2_BASE
   generic map (
      BANDWIDTH => "OPTIMIZED",  -- Jitter programming (OPTIMIZED, HIGH, LOW)
      CLKFBOUT_MULT_F => 8.0,    -- Multiply value for all CLKOUT (2.000-64.000).
      CLKFBOUT_PHASE => 0.0,     -- Phase offset in degrees of CLKFB (-360.000-360.000).
      CLKIN1_PERIOD => 10.0,      -- Input clock period in ns to ps resolution (i.e. 33.333 is 30 MHz).
      -- CLKOUT0_DIVIDE - CLKOUT6_DIVIDE: Divide amount for each CLKOUT (1-128)
      CLKOUT1_DIVIDE => 1,
      CLKOUT2_DIVIDE => 1,
      CLKOUT3_DIVIDE => 1,
      CLKOUT4_DIVIDE => 1,
      CLKOUT5_DIVIDE => 1,
      CLKOUT6_DIVIDE => 1,
      CLKOUT0_DIVIDE_F => 8.0,   -- Divide amount for CLKOUT0 (1.000-128.000).
      -- CLKOUT0_DUTY_CYCLE - CLKOUT6_DUTY_CYCLE: Duty cycle for each CLKOUT (0.01-0.99).
      CLKOUT0_DUTY_CYCLE => 0.5,
      CLKOUT1_DUTY_CYCLE => 0.5,
      CLKOUT2_DUTY_CYCLE => 0.5,
      CLKOUT3_DUTY_CYCLE => 0.5,
      CLKOUT4_DUTY_CYCLE => 0.5,
      CLKOUT5_DUTY_CYCLE => 0.5,
      CLKOUT6_DUTY_CYCLE => 0.5,
      -- CLKOUT0_PHASE - CLKOUT6_PHASE: Phase offset for each CLKOUT (-360.000-360.000).
      CLKOUT0_PHASE => 0.0,
      CLKOUT1_PHASE => 0.0,
      CLKOUT2_PHASE => 0.0,
      CLKOUT3_PHASE => 0.0,
      CLKOUT4_PHASE => 0.0,
      CLKOUT5_PHASE => 0.0,
      CLKOUT6_PHASE => 0.0,
      CLKOUT4_CASCADE => FALSE,  -- Cascade CLKOUT4 counter with CLKOUT6 (FALSE, TRUE)
      DIVCLK_DIVIDE => 1,        -- Master division value (1-106)
      REF_JITTER1 => 0.0,        -- Reference input jitter in UI (0.000-0.999).
      STARTUP_WAIT => FALSE      -- Delays DONE until MMCM is locked (FALSE, TRUE)
   )
   port map (
      -- Clock Outputs: 1-bit (each) output: User configurable clock outputs
      CLKOUT0 => ema_clk,     -- 1-bit output: CLKOUT0
      CLKOUT0B => open,   -- 1-bit output: Inverted CLKOUT0
      CLKOUT1 => open,     -- 1-bit output: CLKOUT1
      CLKOUT1B => open,   -- 1-bit output: Inverted CLKOUT1
      CLKOUT2 => open,     -- 1-bit output: CLKOUT2
      CLKOUT2B => open,   -- 1-bit output: Inverted CLKOUT2
      CLKOUT3 => open,     -- 1-bit output: CLKOUT3
      CLKOUT3B => open,   -- 1-bit output: Inverted CLKOUT3
      CLKOUT4 => open,     -- 1-bit output: CLKOUT4
      CLKOUT5 => open,     -- 1-bit output: CLKOUT5
      CLKOUT6 => open,     -- 1-bit output: CLKOUT6
      -- Feedback Clocks: 1-bit (each) output: Clock feedback ports
      CLKFBOUT => clkfb,   -- 1-bit output: Feedback clock
      CLKFBOUTB => open, -- 1-bit output: Inverted CLKFBOUT
      -- Status Ports: 1-bit (each) output: MMCM status ports
      LOCKED => dcm_locked,       -- 1-bit output: LOCK
      -- Clock Inputs: 1-bit (each) input: Clock input
      CLKIN1 => i_ema_clk,       -- 1-bit input: Clock
      -- Control Ports: 1-bit (each) input: MMCM control ports
      PWRDWN => '0',       -- 1-bit input: Power-down
      RST => dcm_reset,             -- 1-bit input: Reset
      -- Feedback Clocks: 1-bit (each) input: Clock feedback ports
      CLKFBIN => clkfb      -- 1-bit input: Feedback clock
   );

   -- End of MMCME2_BASE_inst instantiation
end generate gen_mmcm;

------------------------------------------------------------------------------
-- EMIF Interface Module: 
------------------------------------------------------------------------------
emififace : EMIFA_iface
   generic map ( 
      DECODE_BITS       => DECODE_BITS, 
      CONFIG            => "MityDSP_L138" 
   )
   port map (
      ema_clk       => ema_clk, 
      i_ema_cs0_n   => i_ema_cs0_n, 
      i_ema_cs2_n   => i_ema_cs2_n, 
      i_ema_cs3_n   => i_ema_cs3_n, 
      i_ema_cs4_n   => i_ema_cs4_n, 
      i_ema_cs5_n   => i_ema_cs5_n, 
      i_ema_oe_n    => i_ema_oe_n, 
      i_ema_we_n    => i_ema_we_n, 
      o_ema_wait_n  => ema_wait_n, 
      t_ema_wait    => t_ema_wait, 
      i_ema_d       => i_ema_d,
      o_ema_d       => o_ema_d, 
      t_ema_d       => t_ema_d, 
      i_ema_a       => i_ema_a, 
      i_ema_ba      => i_ema_ba, 
      i_ema_wen_dqm => i_ema_wen_dqm,  
      i_ema_cas     => i_ema_cas,  
      i_ema_ras     => i_ema_ras,  
      i_ema_sdcke   => i_ema_sdcke,  
      i_ema_rnw     => i_ema_rnw,  
      o_core_be     => be_r,  
      o_core_addr   => addr_r,  
      o_core_cs5    => cs5_r,  
      o_core_cs4    => cs4_r,  
      o_core_edi    => edi_r,  
      i_core_edo5   => edo5,  
      i_core_edo4   => edo4,  
      o_core_rd     => rd_r,  
      o_core_wr     => wr_r  
   );

------------------------------------------------------------------------------
-- Base Module
------------------------------------------------------------------------------
bm : base_module
   generic map (
      CONFIG       => "MityDSP_L138",
      IRQ0_CPU     => INT0_CPU,
      IRQ1_CPU     => INT1_CPU,
      GEN_DCM_RST  => TRUE,
      GEN_DNA_PORT => TRUE
   )
   port map (
      ema_clk         => ema_clk, 
      i_cs            => cs5_r(CORE_BASE_MODULE),
      i_ID            => FPGA_APPLICATION_ID,
      i_version_major => FPGA_VERSION_MAJOR,
      i_version_minor => FPGA_VERSION_MINOR,
      i_year          => FPGA_YEAR,
      i_month         => FPGA_MONTH,
      i_day           => FPGA_DAY,
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo5(CORE_BASE_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_be_r          => be_r,
      i_irq_map       => irq_map,
      o_irq_output    => o_int,
      i_dcm_status    => dcm_status,
      i_dcm_lock      => dcm_locked,
      o_dcm_reset     => dcm_reset
   );

------------------------------------------------------------------------------
-- Base Port Assignments
------------------------------------------------------------------------------
o_nmi_n <= '1';

gen_ema_d_iobs : for i in 0 to 15 generate
begin
   ema_d_iob : iobuf
      port map (
         io => io_ema_d(i),
         o  => i_ema_d(i),
         i  => o_ema_d(i),
         t  => t_ema_d
      );
end generate gen_ema_d_iobs;

io_ema_wait_n <= ema_wait_n when t_ema_wait = '0' else (others=>'Z');

end rtl;