--- Title: lcd_serdes_cl000104.vhd
--- Description:  Optional Serializer to use fewer LCD interface pins
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 09/17/2009
--- Version: 1.00
-------------------------------------------------------------------------------
--- Rev. Date     Eng Description
--- ---- -------- --- ---------------------------------------------------------
--- 1.00 04/16/10 MAW Initial version.
-------------------------------------------------------------------------------

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity lcd_serdes_cl000104 is
   port (
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;  -- ~12.5 MHz
      i_lcd_enb     : in std_logic;
      
      -- LCD control signals (from core???)
      i_lcd_backlit : in std_logic := '1'; -- set to zero to turn off display...
      i_lcd_en      : in std_logic := '1'; -- set to zero to disable clocking the device
      
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk   : out std_logic;
      o_serdes_ctl   : out std_logic;
      o_serdes_b     : out std_logic;
      o_serdes_g     : out std_logic;
      o_serdes_r     : out std_logic
      
   );
end lcd_serdes_cl000104;

--------------------------------------------------------------------------
-- ARCHITECTURE
--------------------------------------------------------------------------
architecture rtl of lcd_serdes_cl000104 is

signal lcd_r, lcd_g, lcd_b, lcd_sync : std_logic_vector(5 downto 0) := "000000";
signal lcd_mux : std_logic_vector(15 downto 0) := "000000";
signal lcd_r_r, lcd_g_r, lcd_b_r : std_logic_vector(5 downto 0) := "000000";
signal lcd_r_sh, lcd_g_sh, lcd_b_sh, lcd_sync_sh : std_logic_vector(6 downto 0) := "0000000";
signal lcd_clk_r : std_logic := '0';
signal serdes_sh : std_logic_vector(6 downto 0) := "0000001";
signal serdes_clk : std_logic := '0';
signal ser_clk : std_logic;
signal lcd_vsync_r : std_logic;
signal lcd_hsync_r : std_logic;
signal lcd_sdat_r : std_logic := '0';
signal pclk : std_logic;
signal lcd_en_m, lcd_en_r : std_logic := '1';

signal ser_fifo_re : std_logic := '0';
signal data_dup_sh : std_logic_vector(1 downto 0) := "01";
signal lcd_clk_sh : std_logic_vector(1 downto 0) := "10";

signal write_addr : std_logic_vector(4 downto 0) := "00000";
signal read_addr  : std_logic_vector(4 downto 0) := "10000";

signal dcm_lock : std_logic;
signal dcm_status : std_logic_vector(7 downto 0);

signal fifo_din, fifo_dout : std_logic_vector(20 downto 0);

begin -- architecture: rtl

DCM_Serdes_Clkgen : DCM_SP
    generic map (
      CLKDV_DIVIDE => 2.0,                   -- CLKDV divide value
                                             -- (1.5,2,2.5,3,3.5,4,4.5,5,5.5,6,6.5,7,7.5,8,9,10,11,12,13,14,15,16).
      CLKFX_DIVIDE => 1,                     -- Divide value - D - (1-32)
      CLKFX_MULTIPLY => 14,                   -- Multiply value - M - (2-32)
      CLKIN_DIVIDE_BY_2 => FALSE,            -- CLKIN divide by two (TRUE/FALSE)
      CLKIN_PERIOD => 160.0,               -- Input clock period specified in nS
      CLKOUT_PHASE_SHIFT => "NONE",          -- Output phase shift (NONE, FIXED, VARIABLE)
      CLK_FEEDBACK => "1X",                  -- Feedback source (NONE, 1X, 2X)
      DESKEW_ADJUST => "SYSTEM_SYNCHRONOUS", -- SYSTEM_SYNCHRNOUS or SOURCE_SYNCHRONOUS
      DFS_FREQUENCY_MODE => "LOW",           -- Unsupported - Do not change value
      DLL_FREQUENCY_MODE => "LOW",           -- Unsupported - Do not change value
      DSS_MODE => "NONE",                    -- Unsupported - Do not change value
      DUTY_CYCLE_CORRECTION => TRUE,         -- Unsupported - Do not change value
      FACTORY_JF => X"c080",                 -- Unsupported - Do not change value
      PHASE_SHIFT => 0,                      -- Amount of fixed phase shift (-255 to 255)
      STARTUP_WAIT => FALSE                  -- Delay config DONE until DCM LOCKED (TRUE/FALSE)
   )
   port map (
      CLK0 => pclk,         -- 1-bit 0 degree clock output
      CLK180 => open,       -- 1-bit 180 degree clock output
      CLK270 => open,       -- 1-bit 270 degree clock output
      CLK2X => open,        -- 1-bit 2X clock frequency clock output
      CLK2X180 => open,     -- 1-bit 2X clock frequency, 180 degree clock output
      CLK90 => open,        -- 1-bit 90 degree clock output
      CLKDV => open,        -- 1-bit Divided clock output
      CLKFX => serdes_clk,     -- 1-bit Digital Frequency Synthesizer output (DFS)
      CLKFX180 => open,     -- 1-bit 180 degree CLKFX output
      LOCKED => dcm_lock,   -- 1-bit DCM Lock Output
      PSDONE => open,       -- 1-bit Phase shift done output
      STATUS => dcm_status, -- 8-bit DCM status output
      CLKFB => pclk,        -- 1-bit Clock feedback input
      CLKIN => i_lcd_pclk,  -- 1-bit Clock input
      DSSEN => open,        -- 1-bit Unsupported
      PSCLK => '0',         -- 1-bit Phase shift clock input
      PSEN => '0',          -- 1-bit Phase shift enable
      PSINCDEC => '0',      -- 1-bit Phase shift increment/decrement input
      RST => '0'            -- 1-bit Active high reset input
   );

-- drive pixel values to zero when "backlight" is shut off ("screen saver")
lcd_mux <= i_lcd_d when i_lcd_backlit = '1' else (others=>'0');

FIFOBLOCK : for i in 0 to 20 generate
begin
RAM32x1FIFO : RAM32X1D
   generic map ( INIT => X"00000000" )
   port map (
       DPO => fifo_dout(i),
       SPO => open,
       A0  => write_addr(0),
       A1  => write_addr(1),
       A2  => write_addr(2),
       A3  => write_addr(3),
       A4  => write_addr(4),
       D   => fifo_din(i),
       DPRA0 => read_addr(0),
       DPRA1 => read_addr(1),
       DPRA2 => read_addr(2),
       DPRA3 => read_addr(3),
       DPRA4 => read_addr(4),
       WCLK  => pclk,
       WE    => '1' );    
end generate;

fifo_din(5 downto 0)   <= lcd_mux(4 downto 0) & '0';
fifo_din(11 downto 6)  <= lcd_mux(10 downto 5);
fifo_din(17 downto 12) <= lcd_mux(15 downto 11) & '0';
fifo_din(18)           <= i_lcd_enb;
fifo_din(19)           <= i_lcd_vsync;
fifo_din(20)           <= i_lcd_hsync;

lcd_b(5 downto 0) <= fifo_dout(5 downto 0);
lcd_g(5 downto 0) <= fifo_dout(11 downto 6);
lcd_r(5 downto 0) <= fifo_dout(17 downto 12);
lcd_sync(2 downto 0) <= fifo_dout(20 downto 18);

writeaddr : process(pclk)
begin
    if rising_edge(pclk) then
        write_addr <= write_addr+'1';
    end if;
end process writeaddr;

readaddr : process(serdes_clk)
begin
    if rising_edge(serdes_clk) then
       if ser_fifo_re='1' then
           read_addr <= read_addr+'1';
       end if;
    end if;
end process readaddr;

o_serdes_clk <= ser_clk;
o_serdes_ctl <= lcd_sync_sh(6);
o_serdes_b <= lcd_b_sh(6);
o_serdes_g <= lcd_g_sh(6);
o_serdes_r <= lcd_r_sh(6);

shiftOut : process (serdes_clk)
begin
   if rising_edge(serdes_clk) then
      lcd_en_m <= i_lcd_en;
      lcd_en_r <= lcd_en_m;
      
      --Duplicate data for 2 SerDes cycles, toggle only dot_clk
      --  Run SerDes at (2 * 7 pixel Clks)
      --  25MHz / 2 cycles => 12.5 MHz dot_clk
      --Use LCD Porches to adjust the refresh rate
      --  With 12.5 MHz dot clock, can get WQVGA at about 60Hz refresh rate
      if serdes_sh(0) = '1' then
         data_dup_sh <= data_dup_sh(0) & data_dup_sh(1);
         lcd_clk_sh <= lcd_clk_sh(0) & lcd_clk_sh(1);
      end if;
      
      if serdes_sh(0) = '1' then
         lcd_clk_r <= lcd_clk_sh(1) and lcd_en_r; -- disable output clock if disabled
      end if;
      ser_fifo_re <= '0';                      --Default
      if serdes_sh(0) = '1' and data_dup_sh(1) = '1' then
         ser_fifo_re <= '1';
         lcd_r_r <= lcd_r;
         lcd_g_r <= lcd_g;
         lcd_b_r <= lcd_b;
         lcd_sdat_r <= lcd_sync(0);
         lcd_vsync_r <= lcd_sync(1);
         lcd_hsync_r <= lcd_sync(2);
      end if;

      serdes_sh <= serdes_sh(5 downto 0) & serdes_sh(6);
      
      if serdes_sh(2) = '1' then
         ser_clk <= '0';
      elsif serdes_sh(5) = '1' then
         ser_clk <= '1';
      end if;

      --The SerDes receiver expects the D(0) first, then D(1)..D(6)
      if serdes_sh(0) = '1' then
         lcd_r_sh(0) <= lcd_clk_r;  -- OLED Pixel Clock
         lcd_r_sh(1) <= lcd_sdat_r; -- OLED Data Enable 
         lcd_r_sh(2) <= lcd_hsync_r;   
         lcd_r_sh(3) <= lcd_b_r(5);    -- blue(7)
         lcd_r_sh(4) <= lcd_b_r(3);    -- blue(5)
         lcd_r_sh(5) <= '0';           -- blue(1)
         lcd_r_sh(6) <= lcd_g_r(5);    -- green(7)

         lcd_g_sh(0) <= lcd_g_r(3);    -- green(5)
         lcd_g_sh(1) <= lcd_g_r(1);    -- green(3)
         lcd_g_sh(2) <= lcd_r_r(3);    -- red(5)
         lcd_g_sh(3) <= lcd_r_r(1);    -- red(3)
         lcd_g_sh(4) <= '0';           -- red(1)
         lcd_g_sh(5) <= lcd_vsync_r;
         lcd_g_sh(6) <= lcd_b_r(0);    -- blue(2)

         lcd_b_sh(0) <= '0';           -- blue(0)
         lcd_b_sh(1) <= lcd_g_r(4);    -- green(6)
         lcd_b_sh(2) <= lcd_g_r(2);    -- green(4)
         lcd_b_sh(3) <= '0';           -- red(0)
         lcd_b_sh(4) <= lcd_r_r(2);    -- red(4)
         lcd_b_sh(5) <= lcd_r_r(4);    -- red(6)
         lcd_b_sh(6) <= '0';           -- green(0)

         lcd_sync_sh(0) <= lcd_g_r(0); -- green(2)
         lcd_sync_sh(1) <= lcd_b_r(1); -- blue(3)
         lcd_sync_sh(2) <= '0';        -- green(1)
         lcd_sync_sh(3) <= lcd_r_r(5); -- red(7)
         lcd_sync_sh(4) <= lcd_b_r(4); -- blue(6)
         lcd_sync_sh(5) <= lcd_b_r(2); -- blue(4)
         lcd_sync_sh(6) <= lcd_r_r(0); -- red(2)
      else
         lcd_r_sh <= lcd_r_sh(5 downto 0) & lcd_r_sh(6);
         lcd_g_sh <= lcd_g_sh(5 downto 0) & lcd_g_sh(6);
         lcd_b_sh <= lcd_b_sh(5 downto 0) & lcd_b_sh(6);
         lcd_sync_sh <= lcd_sync_sh(5 downto 0) & lcd_sync_sh(6);
      end if;
   end if;
end process;

end rtl;
