--- Title:IndustrialIO_top.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 03/7/2010
--- Version: 1.00
---
--- Revision History
---   1.00 - Initial revision

library WORK;
library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;
use WORK.MityDSP_L138_pkg.ALL;

entity IndustrialIO_top is
   generic ( FPGA_FAMILY : string := "Spartan6";  -- Artix7 or Spartan6
             BOARD_REV     : string := "C"; -- "A", "B", or "C"
             DECODE_BITS   : integer range 1 to 9 := 5;
             CAMERA_CONFIG : string := "NONE"; -- "RAW" - route camera I/O hack to RAW Video input interface 
             DISP_CONFIG   : string := "LCD"; -- "LCD", "DVI" or "NONE"
             LCD_TYPE      : string := "SHARP"  -- "SHARP" or "NEC"
   );
   port (
      i_ema_clk	      : in std_logic; -- 100 MHz EMIF clock

      -- DSP EMIFA BUS Interface
      i_ema_cs0_n   : in std_logic; -- RESERVED for SDRAM Controller
      i_ema_cs2_n   : in std_logic; -- ARM core CE space
      i_ema_cs3_n   : in std_logic; -- NAND FLASH space, not used
      i_ema_cs4_n   : in std_logic; -- DSP core CE space
      i_ema_cs5_n   : in std_logic; -- Reserved core space, not used
      i_ema_oe_n    : in std_logic;
      i_ema_we_n    : in std_logic;
      io_ema_wait_n : inout std_logic_vector(1 downto 0);
      io_ema_d      : inout std_logic_vector(15 downto 0);
      i_ema_a       : in std_logic_vector(13 downto 0);
      i_ema_ba      : in std_logic_vector(1 downto 0);
      i_ema_wen_dqm : in std_logic_vector(1 downto 0); 
      i_ema_cas     : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_ras     : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_sdcke   : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_rnw     : in std_logic; -- reserved for SDRAM controller, not used
      
      -- DSP LCD Interface
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;
      i_lcd_enb     : in std_logic;
      
      -- LCD Differential Serializer Interface
      o_disp_a0_p   : out std_logic;
      o_disp_a0_n   : out std_logic;
      o_disp_a1_p   : out std_logic;
      o_disp_a1_n   : out std_logic;
      o_disp_a2_p   : out std_logic;
      o_disp_a2_n   : out std_logic;
      o_disp_a3_p   : out std_logic;
      o_disp_a3_n   : out std_logic;
      o_disp_clkin_p: out std_logic;
      o_disp_clkin_n: out std_logic;
      
      -- Touch screen interface
      o_ts_cs_n      : out std_logic;
      o_ts_clk       : out std_logic;
      o_ts_din       : out std_logic;
      i_ts_dout      : in  std_logic;
      i_ts_busy      : in  std_logic;
      io_ts_PenIRQ_n : inout  std_logic;
      o_disp_cs_n    : out std_logic;
      
      -- DVI controller interface
      io_sda         : inout std_logic;
      o_scl          : out std_logic;
      
      o_dvi_data     : out std_logic_vector(15 downto 0);
      o_dvi_clk      : out std_logic;
      o_dvi_hsync    : out std_logic;
      o_dvi_vsync    : out std_logic;
      o_dvi_de       : out std_logic;

      -- DSP IRQ lines
      o_nmi_n       : out std_logic;
      o_int         : out std_logic_vector(1 downto 0);
      
      -- CAMERA input interface (for camera demo)
      i_cam_data    : in std_logic_vector(9 downto 0);
      i_cam_lv      : in std_logic;
      i_cam_fv      : in std_logic;
      i_cam_pclk    : in std_logic;
      
      -- Raw Video Interface (for camera demo)
      o_vp_clkin     : out std_logic_vector(1 downto 0);
      o_vp_hsync      : out std_logic;
      o_vp_vsync      : out std_logic;
		o_vp_field      : out std_logic;
      o_vp_din        : out std_logic_vector(12 downto 0);
		
		-- camera demo misc signals
		o_pwm           : out std_logic_vector(1 downto 0);
		o_diode         : out std_logic_vector(1 downto 0)
      
  );
end IndustrialIO_top;

--------------------------------------------------------------------------
-- ARCHITECTURE
--------------------------------------------------------------------------
architecture rtl of IndustrialIO_top is

constant FPGA_APPLICATION_ID: std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 00, 8);
constant FPGA_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 4);
constant FPGA_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 00, 4);
constant FPGA_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 10, 5);
constant FPGA_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 04, 4);
constant FPGA_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 20, 5);

constant CORE_BASE_MODULE: integer := 0;

constant CORE_TS_MODULE     : integer := 1;
constant CORE_I2C_MODULE    : integer := 2;
constant CORE_LCD_MODULE    : integer := 3;
constant CORE_DVI_MODULE    : integer := 4;
constant CORE_GPIO_MODULE   : integer := 5;
constant CORE_PWM_MODULE    : integer := 6;

constant CORE_TS_IRQ_LEVEL  : integer := 0;
constant CORE_TS_IRQ_VECTOR : integer := 0;
constant CORE_I2C_IRQ_LEVEL  : integer := 0;
constant CORE_I2C_IRQ_VECTOR : integer := 1;
constant CORE_LCD_IRQ_LEVEL  : integer := 0;
constant CORE_LCD_IRQ_VECTOR : integer := 2;
constant CORE_DVI_IRQ_LEVEL  : integer := 0;
constant CORE_DVI_IRQ_VECTOR : integer := 3;
constant CORE_GPIO_IRQ_LEVEL : integer := 1;
constant CORE_GPIO_IRQ_VECTOR : integer := 0;
constant CORE_PWM_IRQ_LEVEL     : integer := 1;
constant CORE_PWM_IRQ_VECTOR    : integer := 1;

-- clock/reset related signals
signal clk25mhz: std_logic;
signal ema_clk: std_logic;
signal dcm_lock: std_logic;
signal dcm_reset: std_logic;
signal dcm_status: std_logic_vector(2 downto 0);
signal wd_rst: std_logic;

-- EMIFA interface signals
signal ema_d : std_logic_vector(15 downto 0);
signal ema_wait_n : std_logic_vector(1 downto 0) := "11";
signal t_ema_wait : std_logic;
signal t_ema_d : std_logic;
signal be_r : std_logic_vector(1 downto 0);
signal addr_r : std_logic_vector(5 downto 0);
signal arm_cs5_r : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal dsp_cs4_r : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal edi_r : std_logic_vector(15 downto 0);
signal edo_arm : bus16_vector((2**DECODE_BITS)-1 downto 0);
signal edo_dsp : bus16_vector((2**DECODE_BITS)-1 downto 0);
signal rd_r : std_logic;
signal wr_r : std_logic;
signal irq_map  : bus16_vector(1 downto 0) := (others=>(others=>'0')); 

-- LCD Signals
signal serdes_clk : std_logic;
signal serdes_ctl : std_logic;
signal serdes_b : std_logic;
signal serdes_g : std_logic;
signal serdes_r : std_logic; 
signal serdes_debug : std_logic_vector(7 downto 0);
signal lcd_pclk : std_logic;
signal ts_PenIRQ_n : std_logic;
signal t_ts_PenIRQ_n : std_logic;
signal lcd_hrev : std_logic;
signal lcd_dim_pwm : std_logic;
signal lcd_pwr : std_logic;
signal lcd_backlit : std_logic;
signal lcd_vrev : std_logic;
signal lcd_en : std_logic;

-- DVI signals
signal sda, t_sda : std_logic;
signal dvi_data_c0 : std_logic_vector(11 downto 0);
signal dvi_data_c1 : std_logic_vector(11 downto 0);
signal dvi_clk : std_logic;

-- camera interface
signal cam_lv : std_logic := '0';
signal cam_fv : std_logic := '0';
signal cam_data : std_logic_vector(9 downto 0) := (others=>'0');
signal cam_pclk : std_logic := '0';
signal t_cam_sdat : std_logic := '1';
signal cam_sdat : std_logic := '0';
signal h_cnt : std_logic_vector(9 downto 0) := (others=>'0');
signal l_cnt : std_logic_vector(8 downto 0) := (others=>'0');
signal ema_clk_div : std_logic_vector(1 downto 0) := (others=>'0');
signal temp_clk : std_logic;
signal clkfb : std_logic;
signal dcm_clkstopped: std_logic;
signal dcm_locked: std_logic;
signal dcm_clkvalid: std_logic;

component EMIFA_dcm
   port (
      -- Clock in ports
      CLK_IN1           : in     std_logic;
      CLKFB_IN          : in     std_logic;
      -- Clock out ports
      CLK_OUT1          : out    std_logic;
      CLKFB_OUT         : out    std_logic;  
      -- Status and control signals
      RESET             : in     std_logic;
      STATUS            : out    std_logic_vector(2 downto 0);
      INPUT_CLK_STOPPED : out    std_logic;
      LOCKED            : out    std_logic;
      CLK_VALID         : out    std_logic
   );
end component;

begin -- architecture: rtl

--------------------------------------------------------------------------
-- System DCM Instantiation
--------------------------------------------------------------------------
--------------------------------------------------------------------------
-- Clock Buffer Instantiation
--------------------------------------------------------------------------
pclkbuf : bufg port map (O => lcd_pclk, I=> i_lcd_pclk);

--------------------------------------------------------------------------
-- System DCM Instantiation
--------------------------------------------------------------------------
gen_dcm : if FPGA_FAMILY = "Spartan6" generate
begin
emaclkdcm : EMIFA_dcm
   port map (
      CLK_IN1            => i_ema_clk,
      CLKFB_IN           => clkfb,
      CLK_OUT1           => ema_clk,
      CLKFB_OUT           => clkfb,
      RESET              => dcm_reset,
      STATUS             => dcm_status,
      INPUT_CLK_STOPPED  => dcm_clkstopped,
      LOCKED             => dcm_locked,
      CLK_VALID          => dcm_clkvalid
   );
end generate gen_dcm;

gen_mmcm : if FPGA_FAMILY = "Artix7" generate
begin
   -- MMCME2_BASE: Base Mixed Mode Clock Manager
   --              Artix-7
   -- Xilinx HDL Language Template, version 2021.2

   MMCME2_BASE_inst : MMCME2_BASE
   generic map (
      BANDWIDTH => "OPTIMIZED",  -- Jitter programming (OPTIMIZED, HIGH, LOW)
      CLKFBOUT_MULT_F => 8.0,    -- Multiply value for all CLKOUT (2.000-64.000).
      CLKFBOUT_PHASE => 0.0,     -- Phase offset in degrees of CLKFB (-360.000-360.000).
      CLKIN1_PERIOD => 10.0,      -- Input clock period in ns to ps resolution (i.e. 33.333 is 30 MHz).
      -- CLKOUT0_DIVIDE - CLKOUT6_DIVIDE: Divide amount for each CLKOUT (1-128)
      CLKOUT1_DIVIDE => 1,
      CLKOUT2_DIVIDE => 1,
      CLKOUT3_DIVIDE => 1,
      CLKOUT4_DIVIDE => 1,
      CLKOUT5_DIVIDE => 1,
      CLKOUT6_DIVIDE => 1,
      CLKOUT0_DIVIDE_F => 8.0,   -- Divide amount for CLKOUT0 (1.000-128.000).
      -- CLKOUT0_DUTY_CYCLE - CLKOUT6_DUTY_CYCLE: Duty cycle for each CLKOUT (0.01-0.99).
      CLKOUT0_DUTY_CYCLE => 0.5,
      CLKOUT1_DUTY_CYCLE => 0.5,
      CLKOUT2_DUTY_CYCLE => 0.5,
      CLKOUT3_DUTY_CYCLE => 0.5,
      CLKOUT4_DUTY_CYCLE => 0.5,
      CLKOUT5_DUTY_CYCLE => 0.5,
      CLKOUT6_DUTY_CYCLE => 0.5,
      -- CLKOUT0_PHASE - CLKOUT6_PHASE: Phase offset for each CLKOUT (-360.000-360.000).
      CLKOUT0_PHASE => 0.0,
      CLKOUT1_PHASE => 0.0,
      CLKOUT2_PHASE => 0.0,
      CLKOUT3_PHASE => 0.0,
      CLKOUT4_PHASE => 0.0,
      CLKOUT5_PHASE => 0.0,
      CLKOUT6_PHASE => 0.0,
      CLKOUT4_CASCADE => FALSE,  -- Cascade CLKOUT4 counter with CLKOUT6 (FALSE, TRUE)
      DIVCLK_DIVIDE => 1,        -- Master division value (1-106)
      REF_JITTER1 => 0.0,        -- Reference input jitter in UI (0.000-0.999).
      STARTUP_WAIT => FALSE      -- Delays DONE until MMCM is locked (FALSE, TRUE)
   )
   port map (
      -- Clock Outputs: 1-bit (each) output: User configurable clock outputs
      CLKOUT0 => ema_clk,     -- 1-bit output: CLKOUT0
      CLKOUT0B => open,   -- 1-bit output: Inverted CLKOUT0
      CLKOUT1 => open,     -- 1-bit output: CLKOUT1
      CLKOUT1B => open,   -- 1-bit output: Inverted CLKOUT1
      CLKOUT2 => open,     -- 1-bit output: CLKOUT2
      CLKOUT2B => open,   -- 1-bit output: Inverted CLKOUT2
      CLKOUT3 => open,     -- 1-bit output: CLKOUT3
      CLKOUT3B => open,   -- 1-bit output: Inverted CLKOUT3
      CLKOUT4 => open,     -- 1-bit output: CLKOUT4
      CLKOUT5 => open,     -- 1-bit output: CLKOUT5
      CLKOUT6 => open,     -- 1-bit output: CLKOUT6
      -- Feedback Clocks: 1-bit (each) output: Clock feedback ports
      CLKFBOUT => clkfb,   -- 1-bit output: Feedback clock
      CLKFBOUTB => open, -- 1-bit output: Inverted CLKFBOUT
      -- Status Ports: 1-bit (each) output: MMCM status ports
      LOCKED => dcm_locked,       -- 1-bit output: LOCK
      -- Clock Inputs: 1-bit (each) input: Clock input
      CLKIN1 => i_ema_clk,       -- 1-bit input: Clock
      -- Control Ports: 1-bit (each) input: MMCM control ports
      PWRDWN => '0',       -- 1-bit input: Power-down
      RST => dcm_reset,             -- 1-bit input: Reset
      -- Feedback Clocks: 1-bit (each) input: Clock feedback ports
      CLKFBIN => clkfb      -- 1-bit input: Feedback clock
   );

   -- End of MMCME2_BASE_inst instantiation
end generate gen_mmcm;

------------------------------------------------------------------------------
-- EMIF Interface Module: 
------------------------------------------------------------------------------
emififace : EMIFA_iface
    generic map ( 
        DECODE_BITS       => DECODE_BITS, 
        CONFIG            => "MityDSP_L138" 
    )
    port map ( ema_clk   => ema_clk, 
           -- EMIFA signals
           i_ema_cs0_n   => i_ema_cs0_n, 
           i_ema_cs2_n   => i_ema_cs2_n, 
           i_ema_cs3_n   => i_ema_cs3_n, 
           i_ema_cs4_n   => i_ema_cs4_n, 
           i_ema_cs5_n   => i_ema_cs5_n, 
           i_ema_oe_n    => i_ema_oe_n, 
           i_ema_we_n    => i_ema_we_n, 
           o_ema_wait_n  => ema_wait_n, 
           t_ema_wait    => t_ema_wait, 
           i_ema_d       => io_ema_d, 
           o_ema_d       => ema_d, 
           t_ema_d       => t_ema_d, 
           i_ema_a       => i_ema_a, 
           i_ema_ba      => i_ema_ba, 
           i_ema_wen_dqm => i_ema_wen_dqm,  
           i_ema_cas     => i_ema_cas,  
           i_ema_ras     => i_ema_ras,  
           i_ema_sdcke   => i_ema_sdcke,  
           i_ema_rnw     => i_ema_rnw,  
                      
           -- FPGA fabric signals
           o_core_be      => be_r,  
           o_core_addr    => addr_r,  
           o_core_cs5     => arm_cs5_r,  
           o_core_cs4     => dsp_cs4_r,  
           o_core_edi     => edi_r,  
           i_core_edo5    => edo_arm,  
           i_core_edo4    => edo_dsp,  
           o_core_rd      => rd_r,  
           o_core_wr      => wr_r  
    );

------------------------------------------------------------------------------
-- Base Module
------------------------------------------------------------------------------
bm : base_module
   generic map (
      CONFIG => "MityDSP_L138"
      )
   port map (
      ema_clk         => ema_clk, 
      i_cs            => arm_cs5_r(CORE_BASE_MODULE),
      i_ID            => FPGA_APPLICATION_ID,
      i_version_major => FPGA_VERSION_MAJOR,
      i_version_minor => FPGA_VERSION_MINOR,
      i_year          => FPGA_YEAR,
      i_month         => FPGA_MONTH,
      i_day           => FPGA_DAY,
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo_arm(CORE_BASE_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_be_r          => be_r,
            
      o_dcm_reset  => open,
      i_dcm_status => dcm_status,
      i_dcm_lock   => dcm_lock,

      i_irq_map    => irq_map,
      o_irq_output => o_int                  
      
      );

gen_disp : if DISP_CONFIG = "LCD" generate 
begin

lcd_control : lcd_ctlr
   port map (
      ema_clk        => ema_clk,
      i_ABus         => addr_r,
      i_DBus         => edi_r,
      o_DBus         => edo_arm(CORE_LCD_MODULE),
      i_wr_en        => wr_r,
      i_rd_en        => rd_r,
      i_cs           => arm_cs5_r(CORE_LCD_MODULE),
      o_irq          => irq_map(CORE_LCD_IRQ_LEVEL)(CORE_LCD_IRQ_VECTOR),
      i_ilevel       => conv_std_logic_vector(CORE_LCD_IRQ_LEVEL, 2),
      i_ivector      => conv_std_logic_vector(CORE_LCD_IRQ_VECTOR, 4),
      
      o_lcd_hrev     => lcd_hrev,
      o_lcd_dim_pwm  => lcd_dim_pwm,
      o_lcd_pwr      => lcd_pwr,
      o_lcd_backlit  => lcd_backlit,
      o_lcd_vrev     => lcd_vrev,
      o_lcd_en       => lcd_en
                  
      );

gen_sharp : if LCD_TYPE = "SHARP" generate
begin

serdes :  lcd_serdes_cl000099
   port map (
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       => i_lcd_d,
      i_lcd_hsync   => i_lcd_hsync,
      i_lcd_vsync   => i_lcd_vsync,
      i_lcd_mclk    => i_lcd_mclk,
      i_lcd_pclk    => lcd_pclk,
      i_lcd_enb     => i_lcd_enb,
      
      -- LCD control signals (from core???)
      i_lcd_hrev    => lcd_hrev,
      i_lcd_dim_pwm => lcd_dim_pwm,
      i_lcd_pwr     => lcd_pwr,
      i_lcd_backlit => lcd_backlit,
      i_lcd_vrev    => lcd_vrev,
      i_lcd_en      => '1', -- TODO
      
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk  => serdes_clk,
      o_serdes_ctl  => serdes_ctl,
      o_serdes_b    => serdes_b,
      o_serdes_g    => serdes_g,
      o_serdes_r    => serdes_r,
      
      o_debug        => serdes_debug
   );
end generate gen_sharp;

gen_nec : if LCD_TYPE = "NEC" generate
begin

serdes :  lcd_serdes_cl000119
   port map (
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       => i_lcd_d,
      i_lcd_hsync   => i_lcd_hsync,
      i_lcd_vsync   => i_lcd_vsync,
      i_lcd_mclk    => i_lcd_mclk,
      i_lcd_pclk    => lcd_pclk,
      i_lcd_enb     => i_lcd_enb,
      
      -- LCD control signals (from core)
      i_lcd_backlit => lcd_backlit,
      i_lcd_en      => '1', -- TODO
      i_lcd_dim_pwm => lcd_dim_pwm,
      i_lcd_pwr     => lcd_pwr,
      i_lcd_reset   => '0',
      
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk  => serdes_clk,
      o_serdes_ctl  => serdes_ctl,
      o_serdes_b    => serdes_b,
      o_serdes_g    => serdes_g,
      o_serdes_r    => serdes_r
   );
   
end generate gen_nec;

tsctrl : ads7843
   port map (
      emif_clk       => ema_clk,
      i_ABus         => addr_r,
      i_DBus         => edi_r,
      o_DBus         => edo_arm(CORE_TS_MODULE),
      i_wr_en        => wr_r,
      i_rd_en        => rd_r,
      i_cs           => arm_cs5_r(CORE_TS_MODULE),
      o_irq          => irq_map(CORE_TS_IRQ_LEVEL)(CORE_TS_IRQ_VECTOR),
      i_ilevel       => conv_std_logic_vector(CORE_TS_IRQ_LEVEL, 2),
      i_ivector      => conv_std_logic_vector(CORE_TS_IRQ_VECTOR, 4),

      o_ts_cs_n      => o_ts_cs_n,
      o_ts_clk       => o_ts_clk,
      o_ts_din       => o_ts_din,
      i_ts_dout      => i_ts_dout,
      i_ts_busy      => i_ts_busy,
      i_ts_PenIRQ_n  => io_ts_PenIRQ_n,
      o_ts_PenIRQ_n  => ts_PenIRQ_n,
      t_ts_PenIRQ_n  => t_ts_PenIRQ_n,
      o_disp_cs_n    => o_disp_cs_n,
      o_gpio_out     => open
   );

io_ts_PenIRQ_n <= 'Z' when t_ts_PenIRQ_n='1' else ts_PenIRQ_n;

lcd_red: OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      o  => o_disp_a0_p,
      ob => o_disp_a0_n,
      i  => serdes_r );

lcd_green: OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      o  => o_disp_a1_p,
      ob => o_disp_a1_n,
      i  => serdes_g );

lcd_blue: OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      o  => o_disp_a2_p,
      ob => o_disp_a2_n,
      i  => serdes_b );

lcd_ctl: OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      o  => o_disp_a3_p,
      ob => o_disp_a3_n,
      i  => serdes_ctl );

lcd_shclk: OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      o  => o_disp_clkin_p,
      ob => o_disp_clkin_n,
      i  => serdes_clk );

io_sda <= 'Z';
o_scl  <= 'Z';
o_dvi_hsync <= 'Z';
o_dvi_vsync <= 'Z';
o_dvi_de <= 'Z';
o_dvi_clk <= 'Z';

DDR2_NULL : for i in 0 to 15 generate
begin
    o_dvi_data(i) <= 'Z';
end generate DDR2_NULL;

end generate gen_disp;

gen_dvi : if DISP_CONFIG = "DVI" generate 
begin

rev_ab : if BOARD_REV = "A" or BOARD_REV = "B" generate
begin
i2cctrl : i2c 
   Port Map ( 
      clk            => ema_clk,
      i_ABus         => addr_r,
      i_DBus         => edi_r,
      o_DBus         => edo_arm(CORE_I2C_MODULE),
      i_wr_en        => wr_r,
      i_rd_en        => rd_r,
      i_cs           => arm_cs5_r(CORE_I2C_MODULE),
      o_irq          => irq_map(CORE_I2C_IRQ_LEVEL)(CORE_I2C_IRQ_VECTOR),
      i_ilevel       => conv_std_logic_vector(CORE_I2C_IRQ_LEVEL, 2),
      i_ivector      => conv_std_logic_vector(CORE_I2C_IRQ_VECTOR, 4),
      i_sda          => io_sda,
      o_sda          => sda,
      o_sdt          => t_sda,
      o_scl          => o_scl
   );
   
io_sda <= 'Z' when t_sda = '1' else sda;

dvi :  tfp410 
   port map (
      ema_clk        => ema_clk,
      i_ABus         => addr_r,
      i_DBus         => edi_r,
      o_DBus         => edo_arm(CORE_DVI_MODULE),
      i_wr_en        => wr_r,
      i_rd_en        => rd_r,
      i_cs           => arm_cs5_r(CORE_DVI_MODULE),
      o_irq          => irq_map(CORE_DVI_IRQ_LEVEL)(CORE_DVI_IRQ_VECTOR),
      i_ilevel       => conv_std_logic_vector(CORE_DVI_IRQ_LEVEL, 2),
      i_ivector      => conv_std_logic_vector(CORE_DVI_IRQ_VECTOR, 4),

      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       => i_lcd_d,
      i_lcd_hsync   => i_lcd_hsync,
      i_lcd_vsync   => i_lcd_vsync,
      i_lcd_mclk    => i_lcd_mclk,
      i_lcd_pclk    => lcd_pclk,
      i_lcd_enb     => i_lcd_enb,
            
      -- DVI video interface (we assume double data rate interface for pin reduction)
      -- these should be hooked up to the ODDR2 macro...
      o_dvi_data_c0 => dvi_data_c0,
      o_dvi_data_c1 => dvi_data_c1,
      o_dvi_clk     => dvi_clk,
      o_dvi_hsync   => o_dvi_hsync,
      o_dvi_vsync   => o_dvi_vsync,
      o_dvi_de      => o_dvi_de
            
      );

DDR2 : for i in 0 to 11 generate
begin

ODDR2_inst : ODDR2
   generic map (
      DDR_ALIGNMENT => "NONE",
      INIT => '0',
      SRTYPE => "SYNC" )
   port map (
      Q  => o_dvi_data(i),
      C0 => dvi_clk,
      C1 => not dvi_clk,
      CE => '1',
      D0 => dvi_data_c0(i),
      D1 => dvi_data_c1(i),
      R  => '0',
      S  => '0' );

end generate DDR2;

ODDR2_inst_clk : ODDR2
   generic map (
      DDR_ALIGNMENT => "NONE",
      INIT => '0',
      SRTYPE => "SYNC" )
   port map (
      Q  => o_dvi_clk,
      C0 => dvi_clk,
      C1 => not dvi_clk,
      CE => '1',
      D0 => '1',
      D1 => '0',
      R  => '0',
      S  => '0' );

end generate rev_ab;

rev_c : if BOARD_REV = "C" generate
begin

latch_proc : process(lcd_pclk)
begin
    if rising_edge(lcd_pclk) then
      o_dvi_data     <= i_lcd_d;
      o_dvi_hsync    <= i_lcd_hsync;
      o_dvi_vsync    <= i_lcd_vsync;
      o_dvi_de       <= i_lcd_enb;
    end if;
end process latch_proc;

o_dvi_clk <= not lcd_pclk;
io_sda <= 'Z';
o_scl  <= 'Z';

end generate rev_c;

o_ts_cs_n     <= 'Z';
o_ts_clk      <= 'Z';
o_ts_din      <= 'Z';
io_ts_PenIRQ_n <= 'Z';
o_disp_cs_n   <= 'Z';
o_disp_a0_p   <= 'Z';
o_disp_a0_n   <= 'Z';
o_disp_a1_p   <= 'Z';
o_disp_a1_n   <= 'Z';
o_disp_a2_p   <= 'Z';
o_disp_a2_n   <= 'Z';
o_disp_a3_p   <= 'Z';
o_disp_a3_n   <= 'Z';
o_disp_clkin_p   <= 'Z';
o_disp_clkin_n   <= 'Z';

end generate gen_dvi;

gen_camera : if CAMERA_CONFIG = "RAW" generate
begin

cam_clk_bufg : bufg Port Map (I => i_cam_pclk, O => cam_pclk);

latch_cam_data : process(cam_pclk)
begin
    if rising_edge(cam_pclk) then
        cam_data <= i_cam_data;
        cam_lv   <= i_cam_lv;
        cam_fv   <= i_cam_fv;
    end if;
end process;

o_vp_clkin(0)   <= cam_pclk;
o_vp_clkin(1)   <= cam_pclk;
o_vp_hsync    <= cam_lv;
o_vp_vsync    <= cam_fv;
o_vp_field    <= '0';
o_vp_din      <= "000" & cam_data;

pwm1 : pwm 
	generic map (
	    NUM_OUTPUTS => 2
	)
	Port Map (
      emif_clk       => ema_clk,
      i_ABus         => addr_r,
      i_DBus         => edi_r,
      o_DBus         => edo_arm(CORE_PWM_MODULE),
      i_wr_en        => wr_r,
      i_rd_en        => rd_r,
      i_cs           => arm_cs5_r(CORE_PWM_MODULE),
      o_irq          => irq_map(CORE_PWM_IRQ_LEVEL)(CORE_PWM_IRQ_VECTOR),
      i_ilevel       => conv_std_logic_vector(CORE_PWM_IRQ_LEVEL, 2),
      i_ivector      => conv_std_logic_vector(CORE_PWM_IRQ_VECTOR, 4),
	   o_pwm(1 downto 0) => o_pwm,
		o_sync         => open,
		i_sync         => '0'
	);

gpio1 : gpio
    generic map (
      NUM_BANKS       => 1,
      NUM_IO_PER_BANK => 2
	 )
	 Port Map (
      clk             => ema_clk,
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo_arm(CORE_GPIO_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_cs            => arm_cs5_r(CORE_GPIO_MODULE),
      o_irq           => irq_map(CORE_GPIO_IRQ_LEVEL)(CORE_GPIO_IRQ_VECTOR),
      i_ilevel        => conv_std_logic_vector(CORE_GPIO_IRQ_LEVEL, 2),     
      i_ivector       => conv_std_logic_vector(CORE_GPIO_IRQ_VECTOR, 4),   
      i_io            => "00",
      t_io            => open,
      o_io            => o_diode,
      i_initdir       => "11",
      i_initoutval    => "11"	   
	 );

--o_diode <= "11";

end generate gen_camera;

gen_no_camera : if CAMERA_CONFIG /= "RAW" generate
begin

o_vp_clkin(0)   <= 'Z';
o_vp_clkin(1)   <= 'Z';
o_vp_hsync    <= 'Z';
o_vp_vsync    <= 'Z';
o_vp_field    <= 'Z';
o_vp_din      <= (others=>'Z');
o_diode       <= (others=>'Z');
o_pwm         <= (others=>'Z');

end generate gen_no_camera;

------------------------------------------------------------------------------
-- Base Port Assignments
------------------------------------------------------------------------------
o_nmi_n <= '1';
io_ema_d <= ema_d when t_ema_d = '0' else (others=>'Z');
io_ema_wait_n <= ema_wait_n when t_ema_wait = '0' else (others=>'Z');

end rtl;