--	Package File Template
--
--	Purpose: This package defines supplemental types, subtypes, 
--		 constants, and functions 


library IEEE;
use IEEE.STD_LOGIC_1164.all;

package MityDSP_L138_pkg is

-- Declare types
  
   type bus32_vector is
      array(natural range <>) of std_logic_vector(31 downto 0);

   type bus16_vector is
      array(natural range <>) of std_logic_vector(15 downto 0);

   type bus12_vector is
      array(natural range <>) of std_logic_vector(11 downto 0);

   type bus11_vector is
      array(natural range <>) of std_logic_vector(10 downto 0);

   type bus10_vector is
      array(natural range <>) of std_logic_vector(9 downto 0);

   type bus9_vector is
      array(natural range <>) of std_logic_vector(8 downto 0);
      
   type bus8_vector is
      array(natural range <>) of std_logic_vector(7 downto 0);

   type bus5_vector is
      array(natural range <>) of std_logic_vector(4 downto 0);

   type bus4_vector is
      array(natural range <>) of std_logic_vector(3 downto 0);

component core_version
   port (
      clk           : in std_logic;                       -- system clock
      rd            : in std_logic;                       -- read enable
      ID            : in std_logic_vector(7 downto 0);    -- assigned ID number, 0xF0-0xFF are reserved for customers
      version_major : in std_logic_vector(3 downto 0);    -- major version number 1-15
      version_minor : in std_logic_vector(3 downto 0);    -- minor version number 0-15
      year          : in std_logic_vector(4 downto 0);    -- year since 2000
      month         : in std_logic_vector(3 downto 0);    -- month (1-12)
      day           : in std_logic_vector(4 downto 0);    -- day (1-32)
      ilevel        : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      ivector       : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
      o_data        : out std_logic_vector(15 downto 0)
      );
end component;

component EMIFA_iface
   generic ( 
      DECODE_BITS   : integer range 1 to 9 := 5;
      CONFIG        : string := "UNKNOWN"        -- "MityDSP_L138" 
   );
   port (
      ema_clk       : in  std_logic; -- conditioned by a DCM
      
      -- EMIFA direct-connect signals
      i_ema_cs0_n   : in  std_logic; -- Reserved for SDRAM controller, not used
      i_ema_cs2_n   : in  std_logic; -- Reserved core space, not used
      i_ema_cs3_n   : in  std_logic; -- NAND FLASH space, not used
      i_ema_cs4_n   : in  std_logic; -- CPU 1 core space 
      i_ema_cs5_n   : in  std_logic; -- CPU 0 core space
      i_ema_oe_n    : in  std_logic;
      i_ema_we_n    : in  std_logic;
      o_ema_wait_n  : out std_logic_vector(1 downto 0);
      t_ema_wait    : out std_logic;
      i_ema_d       : in  std_logic_vector(15 downto 0);
      o_ema_d       : out std_logic_vector(15 downto 0);
      t_ema_d       : out std_logic;
      i_ema_a       : in  std_logic_vector(13 downto 0);
      i_ema_ba      : in  std_logic_vector(1 downto 0);
      i_ema_wen_dqm : in  std_logic_vector(1 downto 0); 
      i_ema_cas     : in  std_logic; -- reserved for SDRAM controller, not used
      i_ema_ras     : in  std_logic; -- reserved for SDRAM controller, not used
      i_ema_sdcke   : in  std_logic; -- reserved for SDRAM controller, not used
      i_ema_rnw     : in  std_logic; -- reserved for SDRAM controller, not used
      
      -- FPGA core interface signals
      o_core_be     : out std_logic_vector(1 downto 0);
      o_core_addr   : out std_logic_vector(5 downto 0);
      o_core_cs5    : out std_logic_vector((2**DECODE_BITS)-1 downto 0);
      o_core_cs4    : out std_logic_vector((2**DECODE_BITS)-1 downto 0);
      o_core_edi    : out std_logic_vector(15 downto 0);
      i_core_edo5   : in  bus16_vector((2**DECODE_BITS)-1 downto 0);
      i_core_edo4   : in  bus16_vector((2**DECODE_BITS)-1 downto 0);
      o_core_rd     : out std_logic;
      o_core_wr     : out std_logic
   );
end component;

component base_module
   generic (
      CONFIG       : string := "UNKNOWN"; -- "MityDSP_L138", "MityARM_1808"
      IRQ0_CPU     : integer range 0 to 1 := 0; -- IRQ0 CPU enumeration.  On L138, 0 is ARM and 1 is DSP.
      IRQ1_CPU     : integer range 0 to 1 := 1; -- IRQ1 CPU enumeration.  On L138, 0 is ARM and 1 is DSP.
      GEN_DCM_RST  : boolean := FALSE; -- Enables generation of DCM reset logic.  Uses STARTUP_SPARTAN6 primitive.
      GEN_DNA_PORT : boolean := FALSE  -- Enables generation of Device DNA read logic.  Uses DNA_PORT primitive.
   );
   port (
      ema_clk         : in  std_logic;
      i_cs            : in  std_logic;
      i_ID            : in  std_logic_vector(7 downto 0);    -- assigned Application ID number, 0xFF if unassigned
      i_version_major : in  std_logic_vector(3 downto 0);    -- major version number 1-15
      i_version_minor : in  std_logic_vector(3 downto 0);    -- minor version number 0-15
      i_year          : in  std_logic_vector(4 downto 0);    -- year since 2000
      i_month         : in  std_logic_vector(3 downto 0);    -- month (1-12)
      i_day           : in  std_logic_vector(4 downto 0);    -- day (1-32)
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_be_r          : in  std_logic_vector(1 downto 0);
      
      i_irq_map       : in  bus16_vector(1 downto 0) := (others=>(others=>'0'));
      o_irq_output    : out std_logic_vector(1 downto 0);
      
      i_dcm_status    : in  std_logic_vector(2 downto 0);  -- hook to EMIF DCM status lines
      i_dcm_lock      : in  std_logic;                     -- hook to EMIF DCM lock line
      o_dcm_reset     : out std_logic                      -- hook to EMIF DCM reset line
   );
end component;

component EMIFA_dcm
   port (
      -- Clock in ports
      CLK_IN1           : in     std_logic;
      -- Clock out ports
      CLK_OUT1          : out    std_logic;
      -- Status and control signals
      RESET             : in     std_logic;
      STATUS            : out    std_logic_vector(2 downto 0);
      INPUT_CLK_STOPPED : out    std_logic;
      LOCKED            : out    std_logic;
      CLK_VALID         : out    std_logic
   );
end component;

component lcd_ctlr
   port (
      ema_clk         : in std_logic;
      i_cs            : in std_logic;
      i_ABus          : in std_logic_vector(5 downto 0);
      i_DBus          : in std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in std_logic;
      i_rd_en         : in std_logic;
      o_irq           : out std_logic;
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
      
      o_lcd_hrev    : out std_logic;
      o_lcd_dim_pwm : out std_logic;
      o_lcd_pwr     : out std_logic;
      o_lcd_backlit : out std_logic;
      o_lcd_vrev    : out std_logic;
      o_lcd_en      : out std_logic;
      o_lcd_reset_stb : out std_logic                
      );
end component;

component lcd_serdes_cl000099
   port (
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;  -- must be less < 6.25 MHz and >= 5 MHz
      i_lcd_enb     : in std_logic;
      
      -- LCD control signals (from core???)
      i_lcd_hrev    : in std_logic := '0';
      i_lcd_dim_pwm : in std_logic := '1';
      i_lcd_pwr     : in std_logic := '1';
      i_lcd_backlit : in std_logic := '1';
      i_lcd_vrev    : in std_logic := '0';
      i_lcd_en      : in std_logic := '1';
      
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk   : out std_logic;
      o_serdes_ctl   : out std_logic;
      o_serdes_b     : out std_logic;
      o_serdes_g     : out std_logic;
      o_serdes_r     : out std_logic;
      
      o_debug         : out std_logic_vector(7 downto 0)
   );
end component;

component lcd_serdes_cl000104
   port (
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;  -- ~12.5 MHz
      i_lcd_enb     : in std_logic;
      
      -- LCD control signals (from core???)
      i_lcd_backlit : in std_logic := '1'; -- set to zero to turn off display...
      i_lcd_en      : in std_logic := '1'; -- set to zero to disable clocking the device
      
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk   : out std_logic;
      o_serdes_ctl   : out std_logic;
      o_serdes_b     : out std_logic;
      o_serdes_g     : out std_logic;
      o_serdes_r     : out std_logic
   );
end component;

component lcd_serdes_cl000119
   port (
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;  -- ~12.5 MHz
      i_lcd_enb     : in std_logic;
      
      -- LCD control signals (from LCD core)
      i_lcd_backlit : in std_logic := '1'; -- set to zero to turn off display...
      i_lcd_en      : in std_logic := '1'; -- set to zero to disable clocking the device
      i_lcd_dim_pwm : in std_logic := '1';
      i_lcd_pwr     : in std_logic := '1';
      i_lcd_reset   : in std_logic := '1';
      
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk   : out std_logic;
      o_serdes_ctl   : out std_logic;
      o_serdes_b     : out std_logic;
      o_serdes_g     : out std_logic;
      o_serdes_r     : out std_logic
      
   );
end component;

component lcd_serdes_nl8060BC16_01
   generic (
      MAPPING : string := "6BIT" -- "A" or "B" or "6BIT"
   );
   port (
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;  -- ~38 MHz
      i_lcd_enb     : in std_logic;
      
      -- DCM reset
      i_dcm_reset   : std_logic := '0';
            
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk   : out std_logic;
      o_serdes_a0     : out std_logic;
      o_serdes_a1     : out std_logic;
      o_serdes_a2     : out std_logic;
      o_serdes_a3     : out std_logic -- not used in 6 bit mode
      
   );
end component;

component ads7843
   port (
      emif_clk       : in std_logic; -- 100 Mhz EMIF clock
      i_ABus         : in  std_logic_vector(5 downto 0);
      i_DBus         : in  std_logic_vector(15 downto 0);
      o_DBus         : out std_logic_vector(15 downto 0);
      i_wr_en        : in  std_logic;
      i_rd_en        : in  std_logic;
      i_cs           : in  std_logic;	
      o_irq          : out std_logic;
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

      o_ts_cs_n      :out std_logic;
      o_ts_clk       :out std_logic;
      o_ts_din       :out std_logic;
      i_ts_dout      : in std_logic;
      i_ts_busy      : in std_logic;
      i_ts_PenIRQ_n  : in std_logic;
      o_ts_PenIRQ_n  :out std_logic;
      t_ts_PenIRQ_n  :out std_logic;
      o_disp_cs_n    :out std_logic;
      o_gpio_out     :out std_logic_vector(23 downto 0)
   );
end component;

component i2c
   Port ( 
      clk      : in std_logic;
      i_ABus   : in std_logic_vector(5 downto 0);
      i_DBus   : in std_logic_vector(15 downto 0);
      o_DBus   :out std_logic_vector(15 downto 0);
      i_wr_en  : in std_logic;
      i_rd_en  : in std_logic;
      i_cs     : in std_logic;
      o_irq    :out std_logic := '0';
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
      i_sda    : in std_logic;
      o_sda    :out std_logic;
      o_sdt    :out std_logic;
      o_scl    :out std_logic
   );
end component;

component fifo_dram_sync
   generic (
      WIDTH : INTEGER := 16;
      DEPTH_BITS : INTEGER := 5;
      RAM_STYLE_HINT : STRING := "auto"
   );
   port (
      clk      : IN std_logic;
      i_din    : IN std_logic_VECTOR(7 downto 0);
      i_rd_en  : IN std_logic;
      i_wr_en  : IN std_logic;
      o_dout   : OUT std_logic_VECTOR(7 downto 0);
      o_empty  : OUT std_logic; -- one clock delay from read strobe
      o_full   : OUT std_logic; -- one clock delay from write strobe
      o_almost_full : OUT std_logic; -- one clock delay from write stobe, 1 from full
      o_almost_empty : OUT std_logic; -- one clock delay from read strobe, 1 from empty
      i_reset  : IN std_logic := '0';
      o_count  : OUT std_logic_vector(DEPTH_BITS downto 0)
   );
end component;

component tfp410
   port (
      ema_clk         : in std_logic;
      i_cs            : in std_logic;
      i_ABus          : in std_logic_vector(5 downto 0);
      i_DBus          : in std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in std_logic;
      i_rd_en         : in std_logic;
      o_irq           : out std_logic;
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;
      i_lcd_enb     : in std_logic;
            
      -- DVI video interface (we assume double data rate interface for pin reduction)
      -- these should be hooked up to the ODDR2 macro...
      o_dvi_data_c0 : out std_logic_vector(11 downto 0);
      o_dvi_data_c1 : out std_logic_vector(11 downto 0);
      o_dvi_clk     : out std_logic;
      o_dvi_hsync   : out std_logic;
      o_dvi_vsync   : out std_logic;
      o_dvi_de      : out std_logic
   );
end component;

component spi
   Port 
   (  
      emif_clk   : in  std_logic;
      i_ABus   : in std_logic_vector(5 downto 0);
      i_DBus   : in std_logic_vector(15 downto 0);
      o_DBus   :out std_logic_vector(15 downto 0);
      i_wr_en  : in std_logic;
      i_rd_en  : in std_logic;
      i_cs     : in std_logic;
      o_irq    :out std_logic := '0';
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";      
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";
      -- SPI interface signals		
      o_sclk     : out std_logic;                      -- SPI output clock
      o_cs_n     : out std_logic_vector(7 downto 0) := "11111111"; -- SYNC output 
      o_mosi     : out std_logic;                      -- data output 
      i_miso     : in  std_logic := '0';                      -- data input 
		
      -- In/Out FIFO interfaces (NO FIRST WORD FALL THROUGH)
      --    Synchronous on emif_clock
      i_fifo_depth      : in std_logic_vector(2 downto 0) := "110";  -- DEPTH=2^(CONV_INTEGER(i_fifo_depth)+4), MAX 2048
      o_mosi_fifo_wr    : out std_logic := '0';
      o_mosi_fifo_rd    : out std_logic := '0';
      o_mosi_fifo_in    : out std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input
      i_mosi_fifo_out   : in std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data output
      i_mosi_write_cnt  : in std_logic_vector(11 downto 0) := (others=>'0');
      i_mosi_empty      : in std_logic := '0';

      o_miso_fifo_wr    : out std_logic := '0';
      o_miso_fifo_rd    : out std_logic := '0';
      o_miso_fifo_in    : out std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input
      i_miso_fifo_out   : in std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data output
      i_miso_read_cnt   : in std_logic_vector(11 downto 0) := (others=>'0');
      i_miso_empty      : in std_logic := '0';
      
      o_fifo_rst        : out std_logic := '0'

    );
end component;	

component DDC232x4
   port ( 
      -- EMIFA Interface
      i_eclk          : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic;
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";
      -- DDC232 Interface
      i_ddc_clk       : in  std_logic; -- 40 MHz converter clock
      o_ddc_conv      : out std_logic;
      i_ddc_dvalidn1  : in  std_logic;
      i_ddc_dvalidn2  : in  std_logic;
      i_ddc_dvalidn3  : in  std_logic;
      i_ddc_dvalidn4  : in  std_logic;
      o_ddc_dclk      : out std_logic;
      i_ddc_dout      : in  std_logic;
      o_ddc_resetn    : out std_logic;
      o_ddc_clk_cfg   : out std_logic;
      o_ddc_din_cfg1  : out std_logic;
      o_ddc_din_cfg2  : out std_logic;
      o_ddc_din_cfg3  : out std_logic;
      o_ddc_din_cfg4  : out std_logic
   );
end component;		

component pwm is
   Generic
   (
      NUM_OUTPUTS : integer := 2 -- MAX 4
   );
   Port 
   (  
      emif_clk : in  std_logic;
      i_ABus   : in std_logic_vector(5 downto 0);
      i_DBus   : in std_logic_vector(15 downto 0);
      o_DBus   :out std_logic_vector(15 downto 0);
      i_wr_en  : in std_logic;
      i_rd_en  : in std_logic;
      i_cs     : in std_logic;
      o_irq    :out std_logic := '0';
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";      
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";
      o_pwm          : out std_logic_vector(NUM_OUTPUTS-1 downto 0) := (others=>'0');
      o_sync         : out std_logic := '0';
      i_sync         : in  std_logic := '0'
    );                          
end component;

component uart
   Port ( 
      clk        : in  std_logic;
      i_ABus     : in std_logic_vector(5 downto 0);
      i_DBus     : in std_logic_vector(15 downto 0);
      o_DBus     : out std_logic_vector(15 downto 0);
      i_wr_en    : in std_logic;
      i_rd_en    : in std_logic;
      i_cs       : in std_logic;
      o_irq      : out std_logic := '0';
      i_ilevel   : in std_logic_vector(1 downto 0) := "00";      
      i_ivector  : in std_logic_vector(3 downto 0) := "0000";
      i_be       : in std_logic_vector(1 downto 0);

      -- UART LOGIC SIGNALS
      i_rcv          : in std_logic;
      o_xmit         : out std_logic;
      i_xmit_cts     : in std_logic := '0';   -- active low
      o_rcv_cts      : out std_logic;         -- active low
      i_dsr          : in std_logic := '0';   -- active low
      o_dtr          : out std_logic;         -- active low
      i_dcd          : in std_logic := '0';   -- active low
      i_ri           : in std_logic := '0';   -- active low
      o_xmit_enb     : out std_logic          -- transmit enable
   );
end component;

component gpio
   Generic (
      NUM_BANKS       : integer range 1 to 4 := 1;
      NUM_IO_PER_BANK : integer range 1 to 16 := 16
   );
   Port ( 
      clk             : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic := '0';
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";      
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";   
      i_io            : in  std_logic_vector(NUM_BANKS*NUM_IO_PER_BANK-1 downto 0) := (others=>'0');
      t_io            : out std_logic_vector(NUM_BANKS*NUM_IO_PER_BANK-1 downto 0);
      o_io            : out std_logic_vector(NUM_BANKS*NUM_IO_PER_BANK-1 downto 0);
      i_initdir       : in  std_logic_vector(NUM_BANKS*NUM_IO_PER_BANK-1 downto 0) := (others=>'0'); -- 1 is output
      i_initoutval    : in  std_logic_vector(NUM_BANKS*NUM_IO_PER_BANK-1 downto 0) := (others=>'0')  -- default output state
   );
end component;

component SPI_iface
   generic ( 
      DECODE_BITS   : integer range 1 to 9 := 5;
      CONFIG        : string := "UNKNOWN"        -- "MityDSP_L138" 
   );
   port (      
      -- SPI direct-connect signals
      i_spi_cs_n    : in  std_logic;
      i_spi_clk     : in  std_logic;
      i_spi_din     : in  std_logic;
      o_spi_dout    : out std_logic;
      
      -- FPGA core interface signals
      i_core_clk    : in  std_logic;
      o_core_be     : out std_logic_vector(1 downto 0);
      o_core_addr   : out std_logic_vector(5 downto 0);
      o_core_cs     : out std_logic_vector((2**DECODE_BITS)-1 downto 0);
      o_core_edi    : out std_logic_vector(15 downto 0);
      i_core_edo    : in  bus16_vector((2**DECODE_BITS)-1 downto 0);
      o_core_rd     : out std_logic;
      o_core_wr     : out std_logic
   );
end component;

component STEPPER3967
   Port
   ( 
      clk        : in std_logic;
      base_clk   : in std_logic;
      i_ABus     : in std_logic_vector(5 downto 0);
      i_DBus     : in std_logic_vector(15 downto 0);
      o_DBus     :out std_logic_vector(15 downto 0);
      i_wr_en    : in std_logic;
      i_rd_en    : in std_logic;
      i_cs       : in std_logic;
      o_irq      :out std_logic;
      i_ilevel      : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector     : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

      i_ts          : in std_logic_vector(6 downto 0);
      i_estop       : in std_logic := '0';
      i_ms          : in std_logic_vector(1 downto 0) ;   --Can Hard Code Setting
      o_ms          :out std_logic_vector(1 downto 0);    --For SW Control
      i_ms_vld      : in std_logic := '0';                --MS Bits Are Meaningful

      o_reset_n     :out std_logic;
      o_enable_n    :out std_logic;
      o_sleep_n     :out std_logic;
      o_dir         :out std_logic;
      o_step        :out std_logic;
      i_hi_current  : in std_logic_vector(1 downto 0);     --Can Hard Code the bit[s]
      o_hi_current  :out std_logic_vector(1 downto 0);     --For sW Control
      i_hi_curr_vld : in std_logic   --SW Control Available
    );
end component;

component camera_link
   port (
      -- MityDSP Core interface
      emif_clk   : in  std_logic;
      i_ABus     : in  std_logic_vector(5 downto 0);
      i_DBus     : in  std_logic_vector(15 downto 0);
      o_DBus     : out std_logic_vector(15 downto 0);
      i_wr_en    : in  std_logic;
      i_rd_en    : in  std_logic;
      i_cs       : in  std_logic;   
      o_irq      : out std_logic;
      i_ilevel      : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector     : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

      -- CameraLink interface
      o_pwr_down_n : out std_logic;
      i_cam_clk    : in  std_logic;
      i_cam_din    : in  std_logic_vector(27 downto 0);
      o_cc0        : out std_logic;
      o_cc1        : out std_logic;
      o_cc2        : out std_logic;
      o_cc3        : out std_logic;

      -- UPP FIFO write interface
      o_fifo_wr_clk : out std_logic;   
      o_fifo_wr     : out std_logic;
      o_fifo_din    : out std_logic_vector(15 downto 0); -- (15)=frame-sync (14)=line-sync
      o_fifo_rst    : out std_logic
            
   );
end component;

component ADS8329_ext
   generic
   (
      FIFO_DEPTH_TWO_TO_N : integer range 9 to 15 := 10 -- 10 is 1K words
   );
   port 
   ( 
      clk             : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;	
      o_irq           : out std_logic;
      i_ilevel        : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector       : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

      -- External 32-bit First-Word-Fall-Through FIFO Write Ports
      o_fifo_din      : out std_logic_vector(31 downto 0);                    -- fifo's data in
      o_fifo_wr_stb   : out std_logic;                                        -- fifo's write strobe (1 clk wide)
      o_fifo_wr_clk   : out std_logic;                                        -- fifo's write clock
      -- External 32-bit First-Word-Fall-Through FIFO Read Ports
      o_fifo_rd_stb   : out std_logic;                                        -- fifo's read strobe (1 clk wide)
      o_fifo_rd_clk   : out std_logic;                                        -- fifo's read clock
      i_fifo_dout     : in  std_logic_vector(31 downto 0);                    -- fifo's data out
      i_fifo_full     : in  std_logic;                                        -- fifo's full flag
      i_fifo_empty    : in  std_logic;                                        -- fifo's empty
      i_fifo_rd_count : in  std_logic_vector(FIFO_DEPTH_TWO_TO_N-1 downto 0); -- fifo's read count

      -- ADC control / data ports
      i_adc_clk      : in  std_logic;
      i_udata        : in  std_logic_vector(4 downto 0);   -- User Data lines
      i_trigger      : in  std_logic;                      -- external trigger
      o_adc_convst_n : out std_logic;                      -- adc convert start strobe      
      i_adc_eoc      : in  std_logic;                      -- end of convert signal from ADC
      o_adc_sclk     : out std_logic;                      -- ADC data shift clock (1/4 EMIF clock)
      o_adc_sdi      : out std_logic;                      -- ADC SPI input line
      i_adc_sdo      : in  std_logic;                      -- ADC SPI output line
      o_adc_cs_n     : out std_logic;                      -- ADC frame sync line
      i_fifo_re      : in  std_logic;                      -- optional FIFO read enable (for CE 2 access)
      o_fifo_data    : out std_logic_vector(31 downto 0)   -- optional FIFO output enable (for CE 2 access)
    );
end component;
						 
--  type <new_type> is
--    record
--        <type_name>        : std_logic_vector( 7 downto 0);
--        <type_name>        : std_logic;
--    end record;

-- Declare constants

--  constant <constant_name>		: time := <time_unit> ns;
--  constant <constant_name>		: integer := <value>;
 
-- Declare library component entities

-- Declare functions and procedure

--  function <function_name>  (signal <signal_name> : in <type_declaration>) return <type_declaration>;
--  procedure <procedure_name>	(<type_declaration> <constant_name>	: in <type_declaration>);

end MityDSP_L138_pkg;


package body MityDSP_L138_pkg is

-- Example 1
--  function <function_name>  (signal <signal_name> : in <type_declaration>  ) return <type_declaration> is
--    variable <variable_name>     : <type_declaration>;
--  begin
--    <variable_name> := <signal_name> xor <signal_name>);
--    return <variable_name>; 
--  end <function_name>;


-- Example 2
--  function <function_name>  (signal <signal_name> : in <type_declaration>;
--                         signal <signal_name>   : in <type_declaration>  ) return <type_declaration> is
--  begin
--    if (<signal_name> = '1') then
--      return <signal_name>;
--    else
--      return 'Z';
--    end if;
--  end <function_name>;

-- Procedure Example
--  procedure <procedure_name>  (<type_declaration> <constant_name>  : in <type_declaration>) is
--    
--  begin
--    
--  end <procedure_name>;
 
end MityDSP_L138_pkg;
