/**
 * \file  ipc.h
 *
 * \brief No-OS Inter-Processor Communication (IPC) module header file
 *
 *  This module is primarily used to configure IPC, synchronize processors, and
 *  initialize the IPC runtime.
 *
 *  The Ipc header should be included in an application as follows:
 *  \code
 *  #include "ipclite/ipc.h"
 *  \endcode
 */

/* Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */


#ifndef IPC_H_
#define IPC_H_

#if defined (__cplusplus)
extern "C" {
#endif

/* Following typedefs are here to maintain compatibility with IPC package    */
typedef unsigned int   UInt32;
typedef unsigned short UInt16;
typedef unsigned char  UInt8;
typedef int            Int32;
typedef int            Int;
typedef short          Int16;
typedef char           Int8;
typedef unsigned int   UArg;
#ifndef bool
    #define bool int
    #ifndef    false
        #define false ((bool)0)
    #endif
    #ifndef true
        #define true  ((bool)1)
    #endif
#endif
typedef bool            Bool;
typedef void            Void;

/* IPC CPU ID */
#define    IPC_ARM_HOST            1    /*  Name of ARM host CPU    */
#define    IPC_DSP0                2    /*  Name of DSP CPU #1      */

/* IPC INTERRUPT LINES */
#define    IPC_HOST_INTLINE        1    /*  Name of host interrupt  */
#define    IPC_DSP0_INTLINE        2    /*  Name of DSP CPU #1 interrupt */

/* IPC METHOD FOR RECEVING EVENTS */
#define    IPC_INTERRUPT_METHOD    1    /*  Enables IPC interrupt to receive events */
#define IPC_POLLING_METHOD        2    /*  Poll IPC interrupt      */

/* IPC cold boot key */
#define    IPC_BOOT_KEY            0xB007

/* Success and failure codes for the module */

/**
 *  \def    Ipc_S_BUSY
 *  \brief  The resource is still in use
 */
#define Ipc_S_BUSY              2

/**
 *  \def    Ipc_S_ALREADYSETUP
 *  \brief  The module has been already setup
 */
#define Ipc_S_ALREADYSETUP      1

/**
 *  \def    Ipc_S_SUCCESS
 *  \brief  Operation is successful.
 */
#define Ipc_S_SUCCESS           0

/**
 *  \def    Ipc_E_FAIL
 *  \brief  Generic failure.
 */
#define Ipc_E_FAIL             -1

/**
 *  \def    Ipc_E_INVALIDARG
 *  \brief  Argument passed to function is invalid.
 */
#define Ipc_E_INVALIDARG       -2

/**
 *  \def    Ipc_E_MEMORY
 *  \brief  Operation resulted in memory failure.
 */
#define Ipc_E_MEMORY           -3

/**
 *  \def    Ipc_E_ALREADYEXISTS
 *  \brief  The specified entity already exists.
 */
#define Ipc_E_ALREADYEXISTS    -4

/**
 *  \def    Ipc_E_NOTFOUND
 *  \brief  Unable to find the specified entity.
 */
#define Ipc_E_NOTFOUND         -5

/**
 *  \def    Ipc_E_TIMEOUT
 *  \brief  Operation timed out.
 */
#define Ipc_E_TIMEOUT          -6 

/**
 *  \def    Ipc_E_INVALIDSTATE
 *  \brief  Module is not initialized or in an invalid state.
 */
#define Ipc_E_INVALIDSTATE     -7

/**
 *  \def    Ipc_E_OSFAILURE
 *  \brief  A failure occurred in an OS-specific call
 */
#define Ipc_E_OSFAILURE        -8

/**
 *  \def    Ipc_E_RESOURCE
 *  \brief  Specified resource is not available
 */
#define Ipc_E_RESOURCE         -9

/**
 *  \def    Ipc_E_RESTART
 *  \brief  Operation was interrupted. Please restart the operation
 */
#define Ipc_E_RESTART          -10

/**
 *  \def    Ipc_E_NOTREADY
 *  \brief  Operation was not ready.
 */
#define Ipc_E_NOTREADY         -11

/*******************************************************************************
**                          STRUCTURE DEFINITIONS
*******************************************************************************/
 
typedef struct    IPC_notifyEvent
{
    UInt16    eventId;      /* event number                              */
    UInt32     payload;     /* 32-bit value to pass to remote CPU        */
    UInt16     procId;      /* space for "sent from" cpu id (future use) */
    UInt16    lineId;       /* interrupt line                            */
    UInt16    semCnt;       /* semaphore                                 */
    UInt16  key;            /* system key                                */
} IPC_notifyEvent;

/* Structure to configure IPC module for local CPU */
struct IPC_cfg
 {
    UInt16  localProcId;                /* host processor Id                                      */
    UInt16    remoteProcId;             /* remote processor Id                                    */
    UInt16    remoteIntLine;            /* remote CPU interrupt line                              */
    UInt16    ipcRcvMethod;             /* use interrupt or polling to receive                    */
    UInt16    maxEventNum;              /* maximum number of events to be created                 */
    IPC_notifyEvent    *localEvtPtr;    /* IPC use: addr of event data for remote CPU to write to */
    IPC_notifyEvent    *remoteEvtPtr;   /* IPC use: event data to send to remote CPU              */
};


/*******************************************************************************
**                          FUNCTION DEFINITIONS
*******************************************************************************/

/**
 *  \brief      Setup private ipc data structures for local CPU
 *
 *  This functions sets up the IPC private data structure. It should
 *  be called first and only once, unless the return value is #Ipc_E_NOTREADY.
 *  This indicates that the remote CPU is not ready so Ipc_start needs to be
 *  called again. Once ipc sucessfully started, subsequent calls 
 *  returns #Ipc_S_ALREADYSETUP.
 *
 *  \params[in] Address of configuration structure IPC_cfg
 *  \return     Status
 *              - #Ipc_S_SUCCESS: operation was successful
 *              - #Ipc_S_ALREADYSETUP: already successfully called
 *              - #Ipc_E_FAIL: operation failed
 */
Int IPC_init(struct IPC_cfg *IpcObj);

/**
 *  \brief        Register the interrupt for IPC for local CPU
 *  This function should be called after IPC_init and before Notify_start to
 *  set up the interrupt registration for the local CPU for non-OS environment.
 *
 *  If the local CPU is arm, the argument is the AINTC channel number to IRQ.
 *  If the local CPU is C674, the argument is the CPU interrupt number. 
 *  Interrupting via event combiner is not supported.
 *
 *  \return       Status
 *                - #Ipc_S_SUCCESS: operation was successful
 *                - #Ipc_E_FAIL: operation failed, most likely a bad parameter
 */
Int IPC_intRegister(unsigned int val);

#if defined (__cplusplus)
}
#endif /* defined (__cplusplus) */

#endif /* IPC_H_ */
