//*****************************************************************************
//
// imgbutton.h - Prototypes for the image button widget.
//
// Copyright (c) 2008-2010 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 6288 of the Stellaris Graphics Library.
//
//*****************************************************************************

#ifndef __IMGBUTTON_H__
#define __IMGBUTTON_H__

//*****************************************************************************
//
//! \addtogroup imgbutton_api
//! @{
//
//*****************************************************************************

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

//*****************************************************************************
//
//! The structure that describes a image button widget.
//
//*****************************************************************************
typedef struct
{
    //
    //! The generic widget information.
    //
    tWidget sBase;

    //
    //! The style for this widget.  This is a set of flags defined by
    //! IB_STYLE_xxx.
    //
    unsigned int ulStyle;

    //
    //! The color to use for foreground pixels when a 1bpp image or text is in
    //! use.  This value is ignored for all other image bit depths.
    //
    unsigned int ulForegroundColor;

    //
    //! The color to use for background pixels when the button is pressed
    //! and a 1bpp image is in use.  This value is ignored for all other image
    //! bit depths.  If IB_STYLE_FILL is specified, this is also the color that
    //! will be used to fill the widget when it is in the pressed state.
    //
    unsigned int ulPressedColor;

    //
    //! The color to use for background pixels when the button is released
    //! and a 1bpp image is in use.  This value is ignored for all other image
    //! bit depths.  If IB_STYLE_FILL is specified, this is also the color that
    //! will be used to fill the widget when it is in the unpressed state.
    //
    unsigned int ulBackgroundColor;

    //
    //! A pointer to the font used to render the button text, if
    //! IB_STYLE_TEXT is selected.
    //
    const tFont *pFont;

    //
    //! A pointer to the text to draw on this push button, if IB_STYLE_TEXT is
    //! selected.
    //
    const char *pcText;

    //
    //! A pointer to the image to be drawn onto this image button, if
    //! IB_STYLE_IMG is selected.
    //
    const unsigned char *pucImage;

    //
    //! A pointer to the image to be drawn onto this image button when it is
    //! pressed.
    //
    const unsigned char *pucPressImage;

    //
    //! A pointer to the image to be drawn above the background image for
    //! the button.
    //
    const unsigned char *pucKeycapImage;

    //! The number of pixels to move the keycap image horizontally when the
    //! button is drawn in its pressed state.
    //
    short sXOffset;

    //
    //! The number of pixels to move the keycap image vertically when the
    //! button is drawn in its pressed state.
    //
    short sYOffset;

    //
    //! The number of pointer events to delay before starting to auto-repeat,
    //! if IB_STYLE_AUTO_REPEAT is selected.  The amount of time to which this
    //! corresponds is dependent upon the rate at which pointer events are
    //! generated by the pointer driver.
    //
    unsigned short usAutoRepeatDelay;

    //
    //! The number of pointer events between button presses generated by the
    //! auto-repeat function, if IB_STYLE_AUTO_REPEAT is selected.  The amount
    //! of time to which this corresponds is dependent up on the rate at which
    //! pointer events are generated by the pointer driver.
    //
    unsigned short usAutoRepeatRate;

    //
    //! The number of pointer events that have occurred.  This is used when
    //! IB_STYLE_AUTO_REPEAT is selected to generate the auto-repeat events.
    //
    unsigned int ulAutoRepeatCount;

    //
    //! A pointer to the function to be called when the button is pressed.
    //! This is repeatedly called when IB_STYLE_AUTO_REPEAT is selected.
    //
    void (*pfnOnClick)(tWidget *pWidget);
}
tImageButtonWidget;

//*****************************************************************************
//
//! This flag indicates that the image button should be filled.
//
//*****************************************************************************
#define IB_STYLE_FILL           0x00000002

//*****************************************************************************
//
//! This flag indicates that the image button should have text drawn on it.
//
//*****************************************************************************
#define IB_STYLE_TEXT           0x00000004

//*****************************************************************************
//
//! This flag indicates that the background image is to be disabled.
//
//*****************************************************************************
#define IB_STYLE_IMAGE_OFF      0x00000008

//*****************************************************************************
//
//! This flag indicates that the keycap image is to be disabled.
//
//*****************************************************************************
#define IB_STYLE_KEYCAP_OFF     0x00000010

//*****************************************************************************
//
//! This flag indicates that the image button should auto-repeat, generating
//! repeated click events while it is pressed.
//
//*****************************************************************************
#define IB_STYLE_AUTO_REPEAT    0x00000020

//*****************************************************************************
//
//! This flag indicates that the image button is pressed.
//
//*****************************************************************************
#define IB_STYLE_PRESSED        0x00000040

//*****************************************************************************
//
//! This flag indicates that the image button callback should be made when
//! the button is released rather than when it is pressed.  This does not
//! affect the operation of auto repeat buttons.
//
//*****************************************************************************
#define IB_STYLE_RELEASE_NOTIFY 0x00000080

//*****************************************************************************
//
//! Declares an initialized image button widget data structure.
//!
//! \param pParent is a pointer to the parent widget.
//! \param pNext is a pointer to the sibling widget.
//! \param pChild is a pointer to the first child widget.
//! \param pDisplay is a pointer to the display on which to draw the push
//! button.
//! \param lX is the X coordinate of the upper left corner of the image button.
//! \param lY is the Y coordinate of the upper left corner of the image button.
//! \param lWidth is the width of the image button.
//! \param lHeight is the height of the image button.
//! \param ulStyle is the style to be applied to the image button.
//! \param ulForeColor is the color to be used for foreground pixels when
//! a 1bpp image or text is being drawn.  It is ignored for all other image bit
//! depths.
//! \param ulPressColor is the color to be used for foreground pixels when
//! the button is pressed and a 1bpp image is being drawn.  It is ignored for
//! all other image bit depths.
//! \param ulBackColor is the color to be used for background pixels when
//! the button is released and a 1bpp image is being drawn.  It is ignored for
//! all other image bit depths.
//! \param pFont is a pointer to the font to be used to draw text on the button.
//! \param pcText is a pointer to the text to draw on this button.
//! \param pucImage is a pointer to the image to draw on the background of
//! this image button when it is in the released state.
//! \param pucPressImage is a pointer to the image to draw on the background of
//! this image button when it is in the pressed state.
//! \param pucKeycapImage is a pointer to the image to draw as the keycap of the
//! on top of the image button, on top of the background image.
//! \param sXOff is the horizontal offset to apply when drawing the keycap image
//! on the button when in the pressed state.
//! \param sYOff is the vertical offset to apply when drawing the keycap image
//! on the button when in the pressed state.
//! \param usAutoRepeatDelay is the delay before starting auto-repeat.
//! \param usAutoRepeatRate is the rate at which auto-repeat events are
//! generated.
//! \param pfnOnClick is a pointer to the function that is called when the push
//! button is pressed.
//!
//! This macro provides an initialized image button widget data structure,
//! which can be used to construct the widget tree at compile time in global
//! variables (as opposed to run-time via function calls).  This must be
//! assigned to a variable, such as:
//!
//! \verbatim
//!     tImageButtonWidget g_sImageButton = ImageButtonStruct(...);
//! \endverbatim
//!
//! Or, in an array of variables:
//!
//! \verbatim
//!     tImageButtonWidget g_psImageButtons[] =
//!     {
//!         ImageButtonStruct(...),
//!         ImageButtonStruct(...)
//!     };
//! \endverbatim
//!
//! \e ulStyle is the logical OR of the following:
//!
//! - \b #IB_STYLE_TEXT to indicate that text should be drawn on the button.
//! - \b #IB_STYLE_FILL to indicate that the background of the button should
//!   be filled with color.
//! - \b #IB_STYLE_KEYCAP_OFF to indicate that the keycap image should not be
//!   drawn.
//! - \b #IB_STYLE_IMAGE_OFF to indicate that the background image should not be
//!   drawn.
//! - \b #IB_STYLE_AUTO_REPEAT to indicate that auto-repeat should be used.
//! - \b #IB_STYLE_RELEASE_NOTIFY to indicate that the callback should be made
//!   when the button is released.  If absent, the callback is called when the
//!   button is initially pressed.
//!
//! \return Nothing; this is not a function.
//
//*****************************************************************************
#define ImageButtonStruct(pParent, pNext, pChild, pDisplay, lX, lY,          \
                          lWidth, lHeight, ulStyle, ulForeColor,             \
                          ulPressColor, ulBackColor, pFont, pcText, pucImage,\
                          pucPressImage, pucKeycapImage, sXOff, sYOff,       \
                          usAutoRepeatDelay, usAutoRepeatRate, pfnOnClick)   \
        {                                                                    \
            {                                                                \
                sizeof(tImageButtonWidget),                                  \
                (tWidget *)(pParent),                                        \
                (tWidget *)(pNext),                                          \
                (tWidget *)(pChild),                                         \
                pDisplay,                                                    \
                {                                                            \
                    lX,                                                      \
                    lY,                                                      \
                    (lX) + (lWidth) - 1,                                     \
                    (lY) + (lHeight) - 1                                     \
                },                                                           \
                ImageButtonMsgProc                                           \
            },                                                               \
            ulStyle,                                                         \
            ulForeColor,                                                     \
            ulPressColor,                                                    \
            ulBackColor,                                                     \
            pFont,                                                           \
            pcText,                                                          \
            pucImage,                                                        \
            pucPressImage,                                                   \
            pucKeycapImage,                                                  \
            sXOff,                                                           \
            sYOff,                                                           \
            usAutoRepeatDelay,                                               \
            usAutoRepeatRate,                                                \
            0,                                                               \
            pfnOnClick                                                       \
        }

//*****************************************************************************
//
//! Declares an initialized variable containing a image button widget data
//! structure.
//!
//! \param sName is the name of the variable to be declared.
//! \param pParent is a pointer to the parent widget.
//! \param pNext is a pointer to the sibling widget.
//! \param pChild is a pointer to the first child widget.
//! \param pDisplay is a pointer to the display on which to draw the push
//! button.
//! \param lX is the X coordinate of the upper left corner of the image button.
//! \param lY is the Y coordinate of the upper left corner of the image button.
//! \param lWidth is the width of the image button.
//! \param lHeight is the height of the image button.
//! \param ulStyle is the style to be applied to the image button.
//! \param ulForeColor is the color to be used for foreground pixels when
//! a 1bpp image is being drawn.  It is ignored for all other image bit depths.
//! \param ulPressColor is the color to be used for foreground pixels when
//! the button is pressed and a 1bpp image is being drawn.  It is ignored for
//! all other image bit depths.
//! \param ulBackColor is the color to be used for background pixels when
//! the button is released and a 1bpp image is being drawn.  It is ignored for
//! all other image bit depths.
//! \param pFont is a pointer to the font to be used to draw text on the button.
//! \param pcText is a pointer to the text to draw on this button.
//! \param pucImage is a pointer to the image to draw on the background of
//! this image button when it is in the released state.
//! \param pucPressImage is a pointer to the image to draw on the background of
//! this image button when it is in the pressed state.
//! \param pucKeycapImage is a pointer to the image to draw as the keycap of the
//! on top of the image button, on top of the background image.
//! \param sXOff is the horizontal offset to apply when drawing the keycap image
//! on the button when in the pressed state.
//! \param sYOff is the vertical offset to apply when drawing the keycap image
//! on the button when in the pressed state.
//! \param usAutoRepeatDelay is the delay before starting auto-repeat.
//! \param usAutoRepeatRate is the rate at which auto-repeat events are
//! generated.
//! \param pfnOnClick is a pointer to the function that is called when the push
//! button is pressed.
//!
//! This macro provides an initialized image button widget data structure,
//! which can be used to construct the widget tree at compile time in global
//! variables (as opposed to run-time via function calls).
//!
//! \e ulStyle is the logical OR of the following:
//!
//! - \b #IB_STYLE_TEXT to indicate that text should be drawn on the button.
//! - \b #IB_STYLE_FILL to indicate that the background of the button should
//!   be filled with color.
//! - \b #IB_STYLE_KEYCAP_OFF to indicate that the keycap image should not be
//!   drawn.
//! - \b #IB_STYLE_IMAGE_OFF to indicate that the background image should not be
//!   drawn.
//! - \b #IB_STYLE_AUTO_REPEAT to indicate that auto-repeat should be used.
//! - \b #IB_STYLE_RELEASE_NOTIFY to indicate that the callback should be made
//!   when the button is released.  If absent, the callback is called when the
//!   button is initially pressed.
//!
//! \return Nothing; this is not a function.
//
//*****************************************************************************
#define ImageButton(sName, pParent, pNext, pChild, pDisplay, lX, lY,       \
                   lWidth, lHeight, ulStyle, ulForeColor, ulPressColor,    \
                   ulBackColor, pFont, pcText, pucImage, pucPressImage,    \
                   pucKeycapImage, sXOff, sYOff, usAutoRepeatDelay,        \
                   usAutoRepeatRate, pfnOnClick)                           \
        tImageButtonWidget sName =                                         \
            ImageButtonStruct(pParent, pNext, pChild, pDisplay, lX, lY,    \
                              lWidth, lHeight, ulStyle, ulForeColor,       \
                              ulPressColor, ulBackColor, pFont, pcText,    \
                              pucImage, pucPressImage, pucKeycapImage,     \
                              sXOff, sYOff, usAutoRepeatDelay,             \
                              usAutoRepeatRate, pfnOnClick)

//*****************************************************************************
//
//! Sets the auto-repeat delay for a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//! \param usDelay is the number of pointer events before auto-repeat starts.
//!
//! This function sets the delay before auto-repeat begins.  Unpredictable
//! behavior will occur if this is called while the image button is pressed.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonAutoRepeatDelaySet(pWidget, usDelay)                 \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->usAutoRepeatDelay = usDelay;                            \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Disables auto-repeat for a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function disables the auto-repeat behavior of a image button.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonAutoRepeatOff(pWidget)                               \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle &= ~(IB_STYLE_AUTO_REPEAT);                     \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Enables auto-repeat for a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function enables the auto-repeat behavior of a image button.
//! Unpredictable behavior will occur if this is called while the image button
//! is pressed.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonAutoRepeatOn(pWidget)                                \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle |= IB_STYLE_AUTO_REPEAT;                        \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Sets the auto-repeat rate for a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//! \param usRate is the number of pointer events between auto-repeat events.
//!
//! This function sets the rate at which auto-repeat events occur.
//! Unpredictable behavior will occur if this is called while the image button
//! is pressed.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonAutoRepeatRateSet(pWidget, usRate)                   \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->usAutoRepeatRate = usRate;                              \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Sets the function to call when this image button widget is pressed.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//! \param pfnOnClik is a pointer to the function to call.
//!
//! This function sets the function to be called when this image button is
//! pressed.  The supplied function is called when the image button is first
//! pressed, and then repeated while the image button is pressed if auto-repeat
//! is enabled.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonCallbackSet(pWidget, pfnOnClik)                      \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->pfnOnClick = pfnOnClik;                                 \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Sets the fill color of a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param ulColor is the 24-bit RGB color to use to fill the image button.
//!
//! This function changes the color used to fill the background of the image
//! button on the display.  This is a duplicate of ImageButtonBackgroundColorSet
//! which is left for backwards compatibility.  The display is not updated
//! until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonFillColorSet(pWidget, ulColor)                       \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulBackgroundColor = ulColor;                            \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Disables filling of a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function disables the filling of a image button widget.  The display is
//! not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonFillOff(pWidget)                                     \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle &= ~(IB_STYLE_FILL);                            \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Enables filling of a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function enables the filling of a image button widget.  The display is
//! not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonFillOn(pWidget)                                      \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle |= IB_STYLE_FILL;                               \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Enables the background image for an image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function enables the drawing of the background image on an image button
//! widget.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonImageOn(pWidget)                                     \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle &= ~(IB_STYLE_IMAGE_OFF);                       \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Disables the background image for an image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function disables the drawing of the background image on an image
//! button widget.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonImageOff(pWidget)                                    \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle |= IB_STYLE_IMAGE_OFF;                          \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Enables the keycap image for an image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function enables the drawing of the keycap image on an image button
//! widget.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonKeycapOn(pWidget)                                    \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle &= ~(IB_STYLE_KEYCAP_OFF);                      \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Disables the keycap image for an image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function disables the drawing of the keycap image on an image button
//! widget.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonKeycapOff(pWidget)                                   \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle |= IB_STYLE_KEYCAP_OFF;                         \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Changes the image drawn on a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param pImg is a pointer to the image to draw onto the image button.
//!
//! This function changes the image that is drawn onto the background of the
//! image button in its unpressed state.  The display is not updated until the
//! next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonImageSet(pWidget, pImg)                              \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            const unsigned char *pI = pImg;                             \
            pW->pucImage = pI;                                          \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Changes the image drawn on a image button widget when it is pressed.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param pImg is a pointer to the image to draw onto the image button when it
//! is pressed.
//!
//! This function changes the image that is drawn onto the background of the
//! image button in its pressed state.  The display is not updated until the
//! next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonImagePressedSet(pWidget, pImg)                       \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            const unsigned char *pI = pImg;                             \
            pW->pucPressImage = pI;                                     \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Changes the keycap image drawn on a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param pImg is a pointer to the image to draw onto the image button.
//!
//! This function changes the image that is drawn onto the top of the push
//! button.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonImageKeycapSet(pWidget, pImg)                        \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            const unsigned char *pI = (pImg);                           \
            pW->pucKeycapImage = pI;                                    \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Changes the keycap image offset position on an image button widget.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param sX is the signed horizontal position offset for the keycap image
//! when the image button is pressed.  Positive values move the image right.
//! \param sY is the signed vertical position offset for the keycap image
//! when the image button is pressed.  Positive values move the image down.
//!
//! This function changes the position that the keycap image is drawn at when
//! the image button is pressed.  The keycap image is moved iX pixels right and
//! iY pixels down from the center position if the image button is pressed.
//! This feature can be used to support 3D buttons.   The display is not
//! updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonKeycapOffsetSet(pWidget, sX, sY)                     \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->sXOffset = (short)(sX);                                 \
            pW->sYOffset = (short)(sY);                                 \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Sets the color of background pixels when using 1bpp images.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param ulColor is the background color to use.
//!
//! This function changes the color that is used to draw background pixels when
//! a 1bpp image is rendered on the button and the button is in the released
//! state.  The value is ignored for all other image bit depths.  The display
//! is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonBackgroundColorSet(pWidget, ulColor)                 \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulBackgroundColor  = (ulColor);                         \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Sets the color of foreground pixels when using 1bpp images.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param ulColor is the foreground color to use.
//!
//! This function changes the color that is used to draw foreground pixels when
//! a 1bpp image or text string is rendered on the button.  The value is
//! ignored for all other image bit depths.  The display is not updated until
//! the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonForegroundColorSet(pWidget, ulColor)                 \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulForegroundColor  = (ulColor);                         \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Sets the color of foreground pixels when the button is pressed and when
//! using 1bpp images.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param ulColor is the pressed foreground color to use.
//!
//! This function changes the color that is used to draw foreground pixels when
//! a 1bpp image is rendered on the button and the button is in the pressed
//! state.  The value is ignored for all other image bit depths.  The display
//! is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonPressedColorSet(pWidget, ulColor)                    \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulPressedColor  = (ulColor);                            \
        }                                                               \
        while(0)


//*****************************************************************************
//
//! Changes the text drawn on a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to be modified.
//! \param pcTxt is a pointer to the text to draw onto the image button.
//!
//! This function changes the text that is drawn onto the image button.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonTextSet(pWidget, pcTxt)                              \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            const char *pcT = pcTxt;                                    \
            pW->pcText = pcT;                                           \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Disables the text on a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function disables the drawing of text on a image button widget.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonTextOff(pWidget)                                     \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle &= ~(IB_STYLE_TEXT);                            \
        }                                                               \
        while(0)

//*****************************************************************************
//
//! Enables the text on a image button widget.
//!
//! \param pWidget is a pointer to the image button widget to modify.
//!
//! This function enables the drawing of text on a image button widget.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define ImageButtonTextOn(pWidget)                                      \
        do                                                              \
        {                                                               \
            tImageButtonWidget *pW = (tImageButtonWidget *)(pWidget);   \
            pW->ulStyle |= IB_STYLE_TEXT;                               \
        }                                                               \
        while(0)

//*****************************************************************************
//
// Prototypes for the image button widget APIs.
//
//*****************************************************************************
extern int ImageButtonMsgProc(tWidget *pWidget, unsigned int ulMsg,
                               unsigned int ulParam1,
                               unsigned int ulParam2);
extern void ImageButtonInit(tImageButtonWidget *pWidget,
                            const tDisplay *pDisplay, int lX, int lY,
                            int lWidth, int lHeight);

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

#endif // __IMGBUTTON_H__
