--- Title: dual_awg_dac5672.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2012
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 11/27/2012
--- Version: 1.00
--- Revisions: 
---   1.00 - Initial release.

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity dual_awg_dac5672 is
   Port ( 
      clk             : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic := '0';
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";      
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";   
      
      i_cha_clock     : in  std_logic;
      i_cha_data      : in  std_logic_vector(15 downto 0);
      i_cha_enable    : in  std_logic;
      i_cha_start     : in  std_logic;
      o_cha_wait      : out std_logic;
      
      -- DAV5672 interface
      i_dac_clk       : in std_logic;
      o_clka          : out std_logic;
      o_da            : out std_logic_vector(13 downto 0);   
      o_clkb          : out std_logic;
      o_db            : out std_logic_vector(13 downto 0)
      
   );
end dual_awg_dac5672;

architecture rtl of dual_awg_dac5672 is

constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 00, 4);
constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 255,8);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 12, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 11, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 29, 5);

signal version_reg : std_logic_vector(15 downto 0) := x"0000";
signal ver_rd      : std_logic := '0';

signal clk_div     : std_logic_vector(7 downto 0) := x"01";
signal clk_div_m   : std_logic_vector(7 downto 0) := x"01";
signal clke_cnt    : std_logic_vector(7 downto 0) := x"00";
signal clke        : std_logic := '0';
signal mode        : std_logic := '0';
signal da, db      : std_logic_vector(13 downto 0) := (others=>'0');

signal fifo_almost_full : std_logic := '0';
signal fifo_din, fifo_dout : std_logic_vector(14 downto 0) := (others=>'0');
signal fifo_wr : std_logic := '0';
signal upp_wait : std_logic := '0';
attribute keep : string;
signal fifo_full, fifo_empty : std_logic := '0';
attribute keep of fifo_full : signal is "true";
attribute keep of fifo_empty : signal is "true";

COMPONENT fifo_dpram2048x15
  PORT (
    rst : IN STD_LOGIC;
    wr_clk : IN STD_LOGIC;
    rd_clk : IN STD_LOGIC;
    din : IN STD_LOGIC_VECTOR(14 DOWNTO 0);
    wr_en : IN STD_LOGIC;
    rd_en : IN STD_LOGIC;
    dout : OUT STD_LOGIC_VECTOR(14 DOWNTO 0);
    full : OUT STD_LOGIC;
    empty : OUT STD_LOGIC;
    prog_full : OUT STD_LOGIC
  );
end component;

begin

version : core_version
   port map(
      clk           => clk,                  -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_ID,              -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );
      
o_irq <= '0';

reg_read : process(clk)
begin
   if rising_edge(clk) then

       ver_rd <= '0';

       if i_cs='1' then
           case i_ABus is
           
           when "000000" =>
               o_DBus <= version_reg;
               ver_rd <= i_rd_en;
           
           when "000001" => NULL;
               o_DBus(15 downto 8) <= clk_div;
               o_DBus(0) <= mode;
               
           when others => NULL;
           
           end case;
       else
           o_DBus <= (others=>'0');
       end if;   
   end if;
end process reg_read;

DAC_FIFO : fifo_dpram2048x15
  PORT MAP (
    rst       => '0',
    wr_clk    => i_cha_clock,
    rd_clk    => i_dac_clk,
    din       => fifo_din,
    wr_en     => fifo_wr,
    rd_en     => clke,
    dout      => fifo_dout,
    full      => fifo_full,
    empty     => fifo_empty,
    prog_full => fifo_almost_full
  );    

reg_write : process(clk)
begin
   if rising_edge(clk) then
      
       if i_cs='1' and i_wr_en='1' then
           case i_ABus is
           
           when "000001" => 
               clk_div <= i_DBus(15 downto 8);           
               mode    <= i_DBus(0);
                         
           when others => NULL;
           
           end case;
       end if;   
   end if;
end process reg_write;      

out_clock_div : process(i_dac_clk)
begin
    if rising_edge(i_dac_clk) then
        clk_div_m <= clk_div;
		  
        if clke_cnt < clk_div_m then
           clke_cnt <= clke_cnt+'1';
           clke <= '0';
        else
           clke_cnt <= (others=>'0');
           clke <= '1';
        end if;
    end if;
end process out_clock_div;

clock_in_data : process(i_cha_clock)
begin
    if rising_edge(i_cha_clock) then
       fifo_din <= i_cha_data(15) & i_cha_data(13 downto 0);
       fifo_wr  <= i_cha_enable;
       if fifo_almost_full='1' then
           upp_wait <= '1';
       else
           upp_wait <= '0';
       end if;
    end if;
end process clock_in_data;

clock_out_data : process(i_dac_clk)
begin
    if rising_edge(i_dac_clk) then
       -- mode 0 is clock both channels with same data
       if mode='0' then
           da <= fifo_dout(13 downto 0);
           db <= fifo_dout(13 downto 0);
       else -- mode 1, use top bit of FIFO to determine channel
           if fifo_dout(14)='1' then
              db <= fifo_dout(13 downto 0);
           else
              da <= fifo_dout(13 downto 0);
           end if;
       end if;
    end if;
end process clock_out_data;

-- output clock domains, data latched by ADC on rising edge...
-- so we transmit on falling edge...
o_da   <= da;
o_db   <= db;
o_clka <= not i_dac_clk;
o_clkb <= not i_dac_clk;
o_cha_wait <= upp_wait;

end rtl;
